"""
********************************************************************************
* This program was made during an undergraduate summer research project funded
* by the Edinburgh Mathematical Society, for which the author remains
* immensely grateful.

This is a Python module for working with transducers, especially the transducer
monoids relating to automorphisms of the full shift, as discussed in the papers
by Bleak, Cameron and Olukoya. This module aims to work with these monoids and
their transducers in an easy-to-use way. Some of the functionality is that
of the 'AAA' module implemented in GAP by Collin Bleak, Fernando Flores Brito,
Luke Elliott and Feyishayo Olukoya.

This code, in its current form, was authored by Elliott Cawtheray, with all
of the algorithms originating from some of the research papers of Dr. Bleak,
or from discussions with Dr. Bleak.
"""

import networkx as nx
import pygraphviz as pgv
import itertools
import random
import math
import ast
import os

def format_test(transducer):

    # Takes an automaton or transducer saved as a variable in the Python session,
    # and returns either "edgeList", "aaa", or "pgv" depending on the format
    # that the transducer is saved as (or raises an error if format is invalid)

    if type(transducer) == list:
        if type(transducer[0]) == list:
            return "edgeList"
        elif type(transducer[0]) == int:
            return "aaa"
        else:
            raise TypeError("The input must be an automaton or transducer in edgeList, AAA, or pygraphviz graph form")
    elif type(transducer) == pgv.agraph.AGraph:
        return "pgv"
    else:
        raise TypeError("The input must be an automaton or transducer in edgeList, AAA, or pygraphviz graph form")

def format_test_txt(file):

    # Takes an automaton or transducer saved as a txt file in the directory,
    # and returns either "edgeList", "aaa", or "pgv" depending on the format
    # that the transducer is saved as (or raises an error if format is invalid)
    #
    # 'file' must be the string that is the name of the file, NOT including
    # the extension (which in this case would be .txt)
    
    path = file + ".txt"
    try:
        try1 = pgv.AGraph(path)
        return "pgv"
    except pgv.DotError:
        with open(path, "r") as data:
            line = data.readline()
            lineList = list(line)
            linecount = lineList.count(";")
            if linecount in [2, 3]:
                return "edgeList"
            else:
                if line.startswith("Transducer"):
                    line = line.replace("Transducer", "", 1)
                while line[-1] != ";" and line[-1] != "]" and line[-1] != ")":
                    line = line[:-1]
                if line[-1] == ";":
                    line = line[:-1]
                line = ast.literal_eval(line)
                IsAAA = True
                if len(line) != 4:
                    IsAAA = False
                if type(line[0]) != int:
                    IsAAA = False
                if type(line[1]) != int:
                    IsAAA = False
                if type(line[2]) != list:
                    IsAAA = False
                if type(line[3]) != list:
                    IsAAA = False
                if type(line[2][0]) != list:
                    IsAAA = False
                if type(line[3][0]) != list:
                    IsAAA = False
                if type(line[3][0][0]) != list:
                    IsAAA = False
                if IsAAA:
                    return "aaa"
                raise ValueError("txt file does not have a valid format (see Formats section of documentation)")

def edgeList_to_aaa(edgeList):

    # Takes a transducer in edgeList form, and returns the transducer represented
    # in the AAA format, as a 4-length list (with empty outputs for automata)
    #
    # Note that converting an automaton or transducer to AAA form will result
    # in the states being renamed 1, 2, 3, ... upon conversion back
    
    alphabet = get_alphabet_list(edgeList)
    nodeList = get_state_list(edgeList)
    P = []
    L = []
    for node in nodeList:
        listInP = []
        for letter in alphabet:
            nextState = transition_function(edgeList, letter, node)
            nextState = nodeList.index(nextState) + 1
            listInP.append(nextState)
        P.append(listInP)
    if len(edgeList[0]) == 4:
        for node in nodeList:
            listInL = []
            for letter in alphabet:
                outputWord = output_function(edgeList, letter, node)
                outputWordInt = []
                for letter in outputWord:
                    outputWordInt.append(alphabet.index(letter))
                outputWord = outputWordInt
                listInL.append(outputWord)
            L.append(listInL)
    else:
        for node in nodeList:
            listInL = []
            for letter in alphabet:
                listInL.append([])
            L.append(listInL)
    return[len(alphabet),len(alphabet),P,L]

def print_aaa_input(transducer):

    # Takes a transducer, and prints the string that when copied and pasted into
    # the AAA package within GAP, can be used to set it as a variable within
    # that package
    
    aaaForm = edgeList_to_aaa(transducer)
    string = "Transducer("+str(aaaForm[0])+", "+str(aaaForm[1])+", "+str(aaaForm[2])+", "+str(aaaForm[3])+")"
    print(string)
    
def edgeList_to_graph(edgeList):

    # Takes an automaton or transducer in edgeList form, and returns the
    # pygraphviz labelled digraph that represents it

    G = pgv.AGraph(strict=False,directed=True)
    if len(edgeList[0]) == 4:
        for edge in edgeList:
            label1 = str(edge[2])+"|"+str(edge[3])
            G.add_edge(edge[0], edge[1], label=label1)
    elif len(edgeList[0]) == 3:
        for edge in edgeList:
            label1 = str(edge[2])
            G.add_edge(edge[0], edge[1], label=label1)
    else:
        raise ValueError("input is not a 3 or 4-tuple of edges")
    return G

def aaa_to_edgeList(aaaList):

    # Takes a transducer representated in the format used in the GAP AAA module,
    # and returns its edgeList

    return graph_to_edgeList(aaa_to_graph(aaaList))

def aaa_to_graph(transducer):
    
    # Takes an automaton or transducer in aaa form, and returns the
    # pygraphviz labelled digraph that represents it

    G = pgv.AGraph(strict=False,directed=True)
    states = range(len(transducer[2]))
    alphabet = range(len(transducer[2][0]))
    IsTransducer = False
    for node in transducer[3]:
        for element in node:
            if len(element) != 0:
                IsTransducer = True
    if IsTransducer:
        for state in states:
            for letter in alphabet:
                startState = state + 1
                endState = transducer[2][state][letter]
                outputWord = transducer[3][state][letter]
                if len(outputWord) == 1:
                    outputWord = transducer[3][state][letter][0]
                label1 = str(letter) + "|" + str(outputWord)
                G.add_edge(startState, endState, label=label1)
    else:
        for state in states:
            for letter in alphabet:
                startState = state + 1
                endState = transducer[2][state][letter]
                label1 = str(letter)
                G.add_edge(startState, endState, label=label1)
    return G

def graph_to_edgeList(G):

    # Takes a pygraphviz labelled digraph representation of an automaton or
    # transducer and converts it into a Python list of edges, with each edge
    # being recorded as a 4-tuple (start state, end state, input letter, output).
    # Note automata will instead give a list of 3-tuples, with outputs dropped
    
    edges = G.edges()
    edgeList = []
    transducerTest = list(edges[0].attr["label"])
    if "|" in transducerTest:
        while edges != []:
            currentEdge = edges[0]
            label1, label2 = currentEdge.attr["label"].split("|")
            label2 = ast.literal_eval(label2)
            if type(label2) != list:
                label2 = [label2]
            for i in range(len(label2)):
                label2[i] = str(label2[i])
            edgeForEdgeList = [currentEdge[0], currentEdge[1], label1, label2]
            edgeList.append(edgeForEdgeList)
            del edges[0]
    else:
        while edges != []:
            currentEdge = edges[0]
            edgeForEdgeList = [currentEdge[0], currentEdge[1], currentEdge.attr["label"]]
            edgeList.append(edgeForEdgeList)
            del edges[0]
    return edgeList

def graph_to_aaa(G):
    return edgeList_to_aaa(graph_to_edgeList(G))

def txt_to_edgeList(file):

    # Takes a .txt file representing an automata or transducer from the
    # directory, in any format, and returns its edgeList
    #
    # 'file' must be the string that is the name of the file, NOT including
    # the extension (which in this case would be .txt)

    test = format_test_txt(file)
    with open(file + ".txt", "r") as data:
        if test == "edgeList":
            edgeList = []
            try:
                for line in data:
                    line1 = line
                    node1, node2, label1, label2 = line.strip().split("; ")
                    label2 = ast.literal_eval(label2)
                    if type(label2) != list:
                        label2 = [label2]
                    for i in range(len(label2)):
                        label2[i] = str(label2[i])
                    M = [node1,node2,label1,label2]
                    edgeList.append(M)
                return edgeList
            except ValueError:
                node1, node2, label1 = line1.strip().split("; ")
                edgeList.append([node1,node2,label1])
                for line in data:
                    node1, node2, label1 = line.strip().split("; ")
                    edgeList.append([node1,node2,label1])
                return edgeList
        if test == "aaa":
            for line in data:
                if line.startswith("Transducer"):
                    line = line.replace("Transducer", "", 1)
                while line[-1] != ";" and line[-1] != "]" and line[-1] != ")":
                    line = line[:-1]
                if line[-1] == ";":
                    line = line[:-1]
                line = ast.literal_eval(line)
                line = list(line)
                for i in range(len(line[2])):
                    for j in range(len(line[2][i])):
                        line[2][i][j] = str(line[2][i][j])
                return aaa_to_edgeList(line)
        if test == "pgv":
            G = pgv.AGraph(file + ".txt")
            return graph_to_edgeList(G)

def txt_to_aaa(file):
    return edgeList_to_aaa(txt_to_edgeList(file))

def txt_to_graph(file):
    return edgeList_to_graph(txt_to_edgeList(file))

def edgeList_to_txt(edgeList, file):

    # Takes an edgeList representing an automata or transducer and generates
    # a .txt file representing it, which is saved to the directory
    #
    # 'file' must be the string that is to be the name of the file, NOT including
    # the extension (which in this case would be .txt)

    with open(file + ".txt", "w") as data:
        M = []
        if len(edgeList[0]) == 4:
            for edge in edgeList:
                element = str(edge[0])+"; "+str(edge[1])+"; "+str(edge[2])+"; "+str(edge[3])+"\n"
                M.append(element)
            data.writelines(M)
        else:
            for edge in edgeList:
                element = str(edge[0])+"; "+str(edge[1])+"; "+str(edge[2])+"\n"
                M.append(element)
            data.writelines(M)
        
def aaa_to_txt(aaaList, file):

    # Takes a transducer in AAA format, and generates a .txt file representing
    # the transducer, which is saved to the directory
    #
    # 'file' must be the string that is to be the name of the file, NOT including
    # the extension (which in this case would be .txt)

    aaaList = tuple(aaaList)
    with open(file + ".txt", "w") as data:
        data.write(str(aaaList))

def graph_to_txt(G, file):

    # Takes a transducer in pygraphviz graph format, and generates a .txt file
    # representing the transducer, which is saved to the directory
    #
    # 'file' must be the string that is to be the name of the file, NOT including
    # the extension (which in this case would be .txt)
    
    edges = G.edges()
    transducerTest = list(edges[0].attr["label"])
    transducer = graph_to_edgeList(G)
    if "|" in transducerTest:
        IsSynch = IsSynchronous(transducer)
        for edge in edges:
            string = edge.attr['label']
            string = string.replace("'","")
            if IsSynch:
                string = string.replace("[","")
                string = string.replace("]","")
            edge.attr['label'] = string
    else:
        for edge in edges:
            string = edge.attr['label']
            string = string.replace("'","")
            edge.attr['label'] = string
    G.write(file + ".txt")

def anything_to_edgeList(transducer):
    test = format_test(transducer)
    if test == "edgeList":
        return transducer
    if test == "aaa":
        return aaa_to_edgeList(transducer)
    if test == "pgv":
        return graph_to_edgeList(transducer)

def anything_to_aaa(transducer):
    test = format_test(transducer)
    if test == "edgeList":
        return edgeList_to_aaa(transducer)
    if test == "aaa":
        return transducer
    if test == "pgv":
        return graph_to_aaa(transducer)

def anything_to_graph(transducer):
    test = format_test(transducer)
    if test == "edgeList":
        return edgeList_to_graph(transducer)
    if test == "aaa":
        return aaa_to_graph(transducer)
    if test == "pgv":
        return transducer

def draw_graph(transducer, file):

    # Takes a transducer / automaton / graph in edgeList, aaa, or pygraphviz
    # form, or is saved as a .txt file in the directory, and generates a .png
    # file in the directory that is a picture of it.
    #
    # Utilises the function "draw" from the pygraphviz module
    #
    # 'file' must be the string that is to be the name of the file, NOT including
    # the extension (which in this case would be .txt)
    #
    # Defaults to the "dot" layout which I've found works best. This can be
    # changed in the 'settings' menu to the user's preferred layout
    # (this will save a .txt file in the directory specifying the chosen layout,
    # so that it is saved permanently)
    #
    # The (functioning) layouts offered by pygraphviz are:
    # neato, dot, twopi, circo, sfdp
    # See https://graphviz.gitlab.io/pdf/dot.1.pdf for info on these layouts

    if type(transducer) != str:
        G = anything_to_graph(transducer)
    else:
        G = txt_to_graph(transducer)
    edges = G.edges()
    transducerTest = list(edges[0].attr["label"])
    if "|" in transducerTest:
        IsSynch = IsSynchronous(transducer)
        for edge in edges:
            string = edge.attr['label']
            string = string.replace("'","")
            if IsSynch:
                string = string.replace("[","")
                string = string.replace("]","")
            edge.attr['label'] = string
    else:
        for edge in edges:
            string = edge.attr['label']
            string = string.replace("'","")
            edge.attr['label'] = string
    try:
        with open("graphviz_layout.txt", "r") as data:
            layout = data.readline()
    except FileNotFoundError:
        layout = "dot"
    G.layout(prog = layout)
    G.draw(file + ".png")

def get_state_list(transducer):

    # Takes as argument an automaton or transducer saved as a variable in Python
    # and returns a list of its states, ordered lexicographically

    nodeList = []
    for edge in transducer:
        if edge[0] not in nodeList:
            nodeList.append(str(edge[0]))
        if edge[1] not in nodeList:
            nodeList.append(str(edge[1]))
    nodeList.sort()
    return nodeList

def get_alphabet_list(transducer):

    # Takes as argument an automaton or transducer saved as a variable in Python
    # and returns a list of its alphabet, ordered lexicographically

    alphabetList = []
    for edge in transducer:
        if edge[2] not in alphabetList:
            alphabetList.append(edge[2])
    alphabetList.sort()
    return alphabetList
    
def transducer_product(transducer1, transducer2):

    # Takes as argument two transducers, over the same alphabet, that are saved
    # as variables in Python, and returns their transducer product

    alphabet = get_alphabet_list(transducer1)
    if set(alphabet) != set(get_alphabet_list(transducer2)):
        raise ValueError("Transducers must be over the same alphabet to take their product")
    transducer3 = []
    nodeList1 = get_state_list(transducer1)
    nodeList2 = get_state_list(transducer2)
    for node1 in nodeList1:
        for node2 in nodeList2:
            currentNode = "("+str(node1)+","+str(node2)+")"
            for letter in alphabet:
                newState1 = transition_function(transducer1, letter, node1)
                newState2 = transition_function(transducer2, output_function(transducer1, letter, node1), node2)
                transitionNode = "("+str(newState1)+","+str(newState2)+")"
                output = output_function(transducer2, output_function(transducer1, letter, node1), node2)
                transducer3.append([currentNode, transitionNode, letter, output])
    return transducer3

def IsIdentity(transducer):

    # Returns the boolean value of the statement "The transducer induces
    # the identity map"

    transducer = combine_equivalent_states(transducer)
    if len(get_alphabet_list(transducer)) != 1:
        return False
    for edge in transducer:
        for letter in edge[3]:
            if edge[2] != letter:
                return False
    return True

def transition_function_base(transducer, inputLetter, currentNode):
    if type(inputLetter) == list:
        if len(inputLetter) == 1:
            inputLetter = inputLetter[0]
    if inputLetter == [] or inputLetter == '':
        return currentNode
    inputLetter = str(inputLetter)
    currentNode = str(currentNode)
    for edge in transducer:
        if inputLetter == edge[2] and currentNode == edge[0]:
            return edge[1]

def transition_function_rec(transducer, inputWord, currentNode):
    inputWord = list(inputWord)
    currentNode = transition_function_base(transducer, inputWord[0], currentNode)
    del inputWord[0]
    if inputWord == []:
        return currentNode
    return transition_function_rec(transducer, inputWord, currentNode)

def transition_function(transducer, inputWord, currentNode):

    # Returns the final state of a transducer on input 'inputWord', starting at
    # state 'currentNode'. 'inputWord' must be inputted as a list of letters,
    # or can be a string if it is a one-letter word
    
    if inputWord == '' or inputWord == [] or inputWord == ['']:
        return currentNode
    return transition_function_rec(transducer, inputWord, currentNode)

def output_function_base(transducer, inputLetter, currentNode):
    inputLetter = str(inputLetter)
    currentNode = str(currentNode)
    for edge in transducer:
        if inputLetter == edge[2] and currentNode == edge[0]:
            return edge[3]

def output_function_rec(transducer, inputWord, currentNode, outputList):
    inputWord = list(inputWord)
    outputWord = output_function_base(transducer, inputWord[0], currentNode)
    if type(outputWord) != list:
        if not IsSynchronous(transducer):
            raise ValueError("Outputs are not formatted correctly")
        outputWord = [outputWord]
    for letter in outputWord:
        outputList.append(letter)
    currentNode = transition_function_base(transducer, inputWord[0], currentNode)
    del inputWord[0]
    if inputWord == []:
        return outputList
    return output_function_rec(transducer, inputWord, currentNode, outputList)

def output_function(transducer, inputWord, currentNode):

    # Returns the output word of a transducer on input 'inputWord', starting at
    # state 'currentNode'.
    
    outputList = []
    return output_function_rec(transducer, inputWord, currentNode, outputList)

def rename_nodes_int(transducer):

    # Returns the inputted transducer, with states renamed as 1, 2, 3, ...
    # Note that this numbering will be by lexicographic ordering of the names of
    # the states in the input transducer
    
    newEdgeList = []
    nodeList = get_state_list(transducer)
    for edge in transducer:
        for nodePos in range(1,len(nodeList)+1):
            if edge[0] == nodeList[nodePos-1]:
                newEdgeList.append([str(nodePos),edge[1],edge[2],edge[3]])
    newNewEdgeList = []
    for edge in newEdgeList:
        for nodePos in range(1,len(nodeList)+1):
            if edge[1] == nodeList[nodePos-1]:
                newNewEdgeList.append([edge[0],str(nodePos),edge[2],edge[3]])
    return newNewEdgeList

def rename_states_forcing_words(transducer, synchLevel):

    # Returns the inputted transducer, with each state renamed by the set
    # of "synchLevel"-length words that force it

    if len(transducer[0]) != 4:
        raise TypeError("Input must be a transducer.")
    nodeList = get_state_list(transducer)
    newEdgeList = []
    wordNodeAssoc = []
    for node in nodeList:
        forcingWords = forcing_words_for_state(transducer, node, synchLevel)
        wordNodeAssoc.append([forcingWords,node])
    for edge in transducer:
        for wordNode in wordNodeAssoc:
            if edge[0] == wordNode[1]:
                newEdgeList.append([wordNode[0],edge[1],edge[2],edge[3]])
    newNewEdgeList = []
    for edge in newEdgeList:
        for wordNode in wordNodeAssoc:
            if edge[1] == wordNode[1]:
                newNewEdgeList.append([edge[0],str(set(wordNode[0])),edge[2],edge[3]])
    return newNewEdgeList
    
def all_words_of_length_k(alphabet, length):

    # Takes as input a list of letters and an integer, and returns a list of all
    # the words over the alphabet of the list of letters, of length the input
    # integer. Utilises the function "product" from the itertools module

    listoflists = [list(i) for i in list(itertools.product(alphabet, repeat=length))]
    for i in range(len(listoflists)):
        for j in range(len(listoflists[i])):
            listoflists[i][j] = str(listoflists[i][j])
    return listoflists

def forcing_words_for_state(transducer, state, wordlength):

    # Takes as input a transducer, one of its states, and an
    # integer, and returns all the words of length that integer which result
    # in the input state as final state, starting from the input state. In
    # particular, if the transducer is strongly synchronizing at level
    # "wordLength", then this will return a full list of the words that force
    # "state". If transducer is not strongly synch, ValueError is raised

    synchLevel = min_synch_level(transducer)
    if wordlength < synchLevel:
        raise ValueError("Transducer is not synchronizing at level " + str(wordlength))
    nodeList = get_state_list(transducer)
    alphabet = get_alphabet_list(transducer)
    forcingWords = []
    kLengthWords = [list(i) for i in list(itertools.product(alphabet, repeat=wordlength))]
    for word in kLengthWords:
        endState = transition_function(transducer, word, state)
        if endState == state and word not in forcingWords:
            forcingWords.append(word)
    return forcingWords

def dual_level_1(transducer):

    # Takes a synchronous transducer, and returns its
    # level 1 dual

    if not IsSynchronous(transducer):
        raise ValueError("Transducer must be synchronous")
    newEdgeList = []
    for edge in transducer:
        newEdgeList.append([edge[2],edge[3],edge[0],edge[1]])
    return newEdgeList

def dual_level_k(transducer, k):

    # Takes a synchronous transducer, and returns its
    # level 'k' dual

    if k == 1:
        return dual_level_1(transducer)
    if not IsSynchronous(transducer):
        raise ValueError("Transducer must be synchronous")
    newEdgeList = []
    alphabet = get_alphabet_list(transducer)
    nodeList = get_state_list(transducer)
    kLengthWords = all_words_of_length_k(alphabet, k)
    for word in kLengthWords:
        for node in nodeList:
            finalState = transition_function(transducer, word, node)
            outputWord = output_function(transducer, word, node)
            newEdgeList.append([word, outputWord, node, finalState])
    return newEdgeList

def pairwise_eq_relation_to_partition(eqRelation):

    # Takes a list consisting of length-2 lists, treats it as a pairwise
    # -defined equivalence relation, and returns the associated partition
    # as a list of lists
    
    partition = []
    partition2 = []
    eqRelationTrack = [i for i in eqRelation]
    while eqRelation != []:
        boolean = False
        for subset in partition:
            if eqRelation != []:
                if eqRelation[0][0] in subset:
                    subset.add(eqRelation[0][1])
                    del eqRelation[0]
                    boolean = True
                elif eqRelation[0][1] in subset:
                    subset.add(eqRelation[0][0])
                    del eqRelation[0]
                    boolean = True
        if boolean == False:
            subset = set([eqRelation[0][0],eqRelation[0][1]])
            partition.append(subset)
            del eqRelation[0]
    for subset in partition:
        subset = list(subset)
        partition2.append(subset)
    return partition2

def omega_equivalence_partition(transducer):

    # Takes a transducer, and returns the partition (as a list of lists)
    # of its states into omega-equivalence classes
    
    nodeList = get_state_list(transducer)
    alphabet = get_alphabet_list(transducer)
    PossiblyEquivalentPairs = []
    NonEquivalentPairs = []
    random_word = ""
    for node in nodeList:
        random_letter = random.choice(alphabet)
        random_word = random_word + random_letter
    for node1 in nodeList:
        for node2 in nodeList:
            nodePair = set([node1, node2])
            if node1 != node2:
                output1 = output_function(transducer, random_word, node1)
                output2 = output_function(transducer, random_word, node2)
                if output1 == output2:
                    boolean = True
                    for letter in alphabet:
                        output1 = output_function(transducer, letter, node1)
                        output2 = output_function(transducer, letter, node2)
                        if output1 != output2:
                            boolean = False
                    if boolean and nodePair not in PossiblyEquivalentPairs:
                        PossiblyEquivalentPairs.append(nodePair)
                    if not boolean and nodePair not in NonEquivalentPairs:
                        NonEquivalentPairs.append(nodePair)
                elif nodePair not in NonEquivalentPairs:
                    NonEquivalentPairs.append(nodePair)
    PossiblyEquivalentPairs = [list(i) for i in PossiblyEquivalentPairs]
    NonEquivalentPairs = [list(i) for i in NonEquivalentPairs]
    boolean = True
    while boolean:
        boolean = False
        iterlist = [i for i in PossiblyEquivalentPairs]
        for pair in iterlist:
            for letter in alphabet:
                r = transition_function(transducer, letter, pair[0])
                s = transition_function(transducer, letter, pair[1])
                if r != s:
                    if ([r,s] in NonEquivalentPairs or [s,r] in NonEquivalentPairs) and pair in PossiblyEquivalentPairs:
                        PossiblyEquivalentPairs.remove(pair)
                        auxList = [pair[1], pair[0]]
                        if pair not in NonEquivalentPairs and auxList not in NonEquivalentPairs:
                            NonEquivalentPairs.append(pair)
                        boolean = True
    partition = pairwise_eq_relation_to_partition(PossiblyEquivalentPairs)
    for node in nodeList:
        InPartition = False
        for subset in partition:
            if node in subset:
                InPartition = True
        if not InPartition:
            partition.append([node])
    return partition

def combine_states(transducer, partition):

    # Takes an automaton or transducer, and returns the automaton or transducer
    # obtained by identifying states as governed by the list 'partition'
    # that represents a partition of the states of 'automaton'

    newEdgeList = []
    statedict = dict()
    for subset in partition:
        for element in subset:
            statedict[element] = subset[0]
    for edge in transducer:
        tupl = (statedict[edge[0]], statedict[edge[1]], edge[2], edge[3])
        if tupl not in newEdgeList:
            newEdgeList.append(tupl)
    newEdgeList.sort()
    for i in range(len(newEdgeList)):
        newEdgeList[i] = list(newEdgeList[i])
    return newEdgeList

def combine_equivalent_states(transducer):

    # Takes a transducer, and returns the weakly-minimal transducer
    # obtained by identifying all omega-equivalent states with each other

    return combine_states(transducer, omega_equivalence_partition(transducer))

def synchronizing_sequence(automaton, term):
    
    # Takes an automaton (or a transducer, and converts to underlying automaton)
    # and an integer 'term', and returns the 'term'th term of the
    # synchronizing sequence of the automaton

    if len(automaton[0]) == 4:
        automaton = transducer_to_automaton(automaton)
    alphabet = get_alphabet_list(automaton)
    states = get_state_list(automaton)
    statedict = dict()
    for state in states:
        statedict[state] = state
    for i in range(term):
        states = get_state_list(automaton)
        partition = []
        classifyingDict = dict()
        for state in states:
            transitions = []
            for letter in alphabet:
                transition = transition_function(automaton, letter, state)
                transitions.append(statedict[transition])
            transitions = str(transitions)
            if transitions not in classifyingDict:
                classifyingDict[transitions] = [state]
            else:
                classifyingDict[transitions].append(state)
        for transition in classifyingDict:
            partition.append(classifyingDict[transition])
        for subset in partition:
            for element in subset:
                statedict[element] = subset[0]
        newEdgeList = []
        for subset in partition:
            for letter in alphabet:
                edge = [subset[0], statedict[transition_function(automaton, letter, subset[0])], letter]
                newEdgeList.append(edge)
        automaton = newEdgeList
    return automaton

def synch_seq_identified_states(automaton, term):
    
    # Returns the dict used by synchronizing sequence, which has the states
    # of the automaton as its keys, and the associated values as the state
    # of the term in the sequence with which the key has been identified with

    if len(automaton[0]) == 4:
        automaton = transducer_to_automaton(automaton)
    alphabet = get_alphabet_list(automaton)
    states = get_state_list(automaton)
    origstates = states
    statedict = dict()
    for state in states:
        statedict[state] = state
    for i in range(term):
        states = get_state_list(automaton)
        partition = []
        classifyingDict = dict()
        for state in states:
            transitions = []
            for letter in alphabet:
                transition = transition_function(automaton, letter, state)
                transitions.append(statedict[transition])
            transitions = str(transitions)
            if transitions not in classifyingDict:
                classifyingDict[transitions] = [state]
            else:
                classifyingDict[transitions].append(state)
        for transition in classifyingDict:
            partition.append(classifyingDict[transition])
        for subset in partition:
            for element in subset:
                statedict[element] = subset[0]
        newEdgeList = []
        for subset in partition:
            for letter in alphabet:
                edge = [subset[0], statedict[transition_function(automaton, letter, subset[0])], letter]
                newEdgeList.append(edge)
        automaton = newEdgeList
    states = get_state_list(automaton)
    returndict = dict()
    for state in origstates:
        value = state
        while value not in states:
            value = statedict[value]
        returndict[state] = value
    return returndict

def IsStronglySynchronizing(transducer):
    
    # Returns 'True' if its input is a strongly synchronizing automaton /
    # transducer and returns 'False' if its input is not strongly synchronizing
    
    looping = True
    i = 0
    if len(transducer[0]) == 4:
        automaton = transducer_to_automaton(transducer)
    else:
        automaton = transducer
    currentstates = get_state_list(automaton)
    while looping:
        i = i + 1
        laststates = currentstates
        synchSeq = synchronizing_sequence(automaton, i)
        currentstates = get_state_list(synchSeq)
        if len(currentstates) == len(laststates):
            looping = False
    if len(currentstates) == 1:
        return True
    else:
        return False

def min_synch_level(transducer):

    # Returns the minimal integer 'k' such that the input automaton / transducer
    # is synchronizing at level k. If the automaton / transducer is not strongly
    # synchronizing, ValueError is raised

    looping = True
    i = 0
    if len(transducer[0]) == 4:
        automaton = transducer_to_automaton(transducer)
    else:
        automaton = transducer
    currentstates = get_state_list(automaton)
    while looping:
        i = i + 1
        laststates = currentstates
        synchSeq = synchronizing_sequence(automaton, i)
        currentstates = get_state_list(synchSeq)
        if len(currentstates) == len(laststates):
            looping = False
    if len(currentstates) == 1:
        return i-1
    else:
        raise ValueError("Automaton / transducer is not strongly synchronizing.")

def take_core(transducer):

    # Takes a strongly synchronizing automaton / transducer, and returns its
    # core (the automaton / transducer consisting only of the states that are
    # reachable for some word longer than any arbitrary finite length)

    alphabet = get_alphabet_list(transducer)
    if not IsStronglySynchronizing(transducer):
        raise ValueError("Automaton / transducer is not strongly synchronizing.")
    synchLevel = min_synch_level(transducer)
    allWords = all_words_of_length_k(alphabet, synchLevel)
    coreEdgeList = []
    coreStates = []
    for word in allWords:
        finalState = transition_function(transducer, word, transducer[0][0])
        coreStates.append(finalState)
    for state in coreStates:
        for letter in alphabet:
            finalState = transition_function(transducer, letter, state)
            if len(transducer[0]) == 4:
                output = output_function(transducer, letter, state)
                coreEdgeList.append([state, finalState, letter, output])
            else:
                coreEdgeList.append([state, finalState, letter])
    return coreEdgeList
    
def transducer_to_automaton(transducer):

    # Takes a transducer, and returns its underlying automaton
    # (i.e, with outputs ignored)

    return [edge[0:3] for edge in transducer]

def de_bruijn_automaton(n,m):

    # Takes as input two integers "n" and "m", and returns
    # the de Bruijn automaton G(n,m)
    
    alphabet = list(range(n))
    nodeList = all_words_of_length_k(alphabet,m)
    edgeList = []
    for node1 in nodeList:
        for node2 in nodeList:
            boolean = True
            for i in range(1,m):
                if node1[i] != node2[i-1]:
                    boolean = False
            if boolean:
                edgeList.append([str(node1),str(node2),str(node2[m-1])])
    return edgeList
    
def folded_de_bruijn(deBruijnAutomaton, partition):

    # Takes a de Bruijn automaton, and a partition of its states (as a list/tuple
    # of lists/tuples) and returns the corresponding folded automaton

    # DEVELOPER NOTE: There is no check for whether the partition gives a
    # vaild folding

    alphabet = get_alphabet_list(deBruijnAutomaton)
    nodeList = []
    for subset in partition:
        nodeList.append(subset[0])
    newEdgeList = []
    for node in nodeList:
        for letter in alphabet:
            endNode = transition_function(deBruijnAutomaton, letter, node)
            for subset in partition:
                if endNode in subset:
                    endNode = subset[0]
            newEdgeList.append(["["+str(node)+"]","["+str(endNode)+"]",letter])
    return newEdgeList

def IsSynchronous(transducer):

    # For a correctly formatted transducer, returns 'True' if it synchronous,
    # 'False' if it is not
    
    alphabet = get_alphabet_list(transducer)
    for edge in transducer:
        if len(edge) != 4:
            raise ValueError("Must be a transducer for IsSynchronous")
        if type(edge[3]) == list:
            if len(edge[3]) != 1:
                return False
        elif type(edge[3]) == str or type(edge[3]) == int:
            if str(edge[3]) not in alphabet:
                return False
        else:
            raise ValueError("Output is not formatted correctly.")
    return True

def IsInvertible(transducer):

    # Currently only works for synchronous transducer

    if not IsSynchronous(transducer):
        raise ValueError("IsInvertible currently only supports synchronous transducers")
    states = get_state_list(transducer)
    alphabet = get_alphabet_list(transducer)
    for state in states:
        testlist = [output_function(transducer, letter, state)[0] for letter in alphabet]
        if len(set(testlist)) != len(testlist):
            return False
    return True

def invert(transducer):

    # Currently only works for synchronous transducers

    if not IsSynchronous(transducer):
        raise ValueError("Transducer is not synchronous")
    if not IsInvertible(transducer):
        raise ValueError("Transducer is not invertible")
    newEdgeList = []
    for edge in transducer:
        output = edge[3]
        if type(output) == list:
            output = output[0]
        output = str(output)
        newEdgeList.append([edge[0],edge[1],output,edge[2]])
    return newEdgeList

def IsBiSynchronizing(transducer):

    # Currently only works for synchronous transducer

    if not IsSynchronous(transducer):
        raise ValueError("IsBiSynchronizing currently only supports synchronous transducers")
    if not IsStronglySynchronizing(transducer):
        return False
    if not IsInvertible(transducer):
        return False
    if not IsStronglySynchronizing(invert(transducer)):
        return False
    return True

def IsWeaklyMinimal(transducer):
    if len(omega_equivalence_partition(transducer)) == len(get_state_list(transducer)):
        return True
    return False

def IsCore(transducer):
    if not IsStronglySynchronizing(transducer):
        return False
    coretransducer = take_core(transducer)
    states = get_state_list(transducer)
    corestates = get_state_list(coretransducer)
    if len(states) == len(corestates):
        return True
    return False

def IsInHn(transducer):
    if IsCore(transducer) and IsSynchronous(transducer) and IsInvertible(transducer) and IsBiSynchronizing(transducer):
        if IsWeaklyMinimal(transducer):
            return True
        else:
            return "Half true"
    return False

def IsValidAutomorphismForAutomaton(automaton, automorphism):

    # Returns True if 'automorphism' is a valid graph automorphism of the
    # underlying graph of the automaton 'automaton'
    #
    # Note: the automorphism must be inputted as a 2-tuple (V, E),
    # where V is a dict that permutes the vertices, and E is a dict that
    # permutes the edges (and the edges are treated as 3-tuples (p,x,q),
    # going from vertex p to vertex q with label x)
    
    testlist = [automorphism[0][key] for key in automorphism[0]]
    if len(testlist) != len(set(testlist)):
        return False
    testlist = [automorphism[1][key] for key in automorphism[1]]
    if len(testlist) != len(set(testlist)):
        return False
    nodes = get_state_list(automaton)
    edges = [(edge[0],edge[2],edge[1]) for edge in automaton]
    for key in automorphism[0]:
        if key not in nodes:
            return False
        if automorphism[0][key] not in nodes:
            return False
    for key in automorphism[1]:
        if key not in edges:
            return False
        if automorphism[1][key] not in edges:
            return False
    for edge in edges:
        if automorphism[0][edge[0]] != automorphism[1][edge][0] or automorphism[0][edge[2]] != automorphism[1][edge][2]:
            return False
    return True

def transducer_from_graph_automorphism(automaton, automorphism):

    # Takes an automaton, and an automorphism of its underlying graph,
    # and constructs a transducer by 'gluing' the automaton along itself
    # according to the automorphism in order to obtain outputs
    #
    # Note: the automorphism must be inputted as a 2-tuple (V, E),
    # where V is a dict that permutes the vertices, and E is a dict that
    # permutes the edges (and the edges are treated as 3-tuples (p,x,q),
    # going from vertex p to vertex q with label x)

    if not IsValidAutomorphismForAutomaton(automaton, automorphism):
        raise ValueError("Not a valid graph automorphism")
    nodes = get_state_list(automaton)
    edges = [(edge[0],edge[2],edge[1]) for edge in automaton]
    newEdgeList = []
    for edge in edges:
        newEdge = automorphism[1][edge]
        newEdgeList.append([edge[0],edge[2],edge[1],newEdge[1]])
    return newEdgeList

def decompose_Hn_element(transducer):
    
    # Takes a transducer that is a representative of an element of the group H_n
    # and runs the decomposition algorithm of the one-sided case paper on it,
    # returning a list of finite order transducers whose product in H_n is the
    # input transducer, where the order of the product is given by the order of
    # the list, i.e. the first element of the list will be a single state
    # transducer, and the final element of the list will be the first transducer
    # found by the algorithm
    
    states = get_state_list(transducer)
    alphabet = get_alphabet_list(transducer)
    if not IsInHn(transducer):
        raise ValueError("Transducer is not in H_n")
    TorsionElementsList = []
    while len(states) > 1:
        A = transducer_to_automaton(transducer)
        B = transducer_to_automaton(invert(transducer))
        A1 = synchronizing_sequence(A,1)
        statedict = synch_seq_identified_states(A,1)
        identified = False
        i = 0
        pair = ""
        while not identified:
            j = 0
            p = list(statedict.keys())[i]
            for forloop in statedict:
                q = list(statedict.keys())[j]
                if statedict[p] == statedict[q] and pair == "" and i != j:
                    pair = [p,q]
                    identified = True
                j = j + 1
            i = i + 1
        p = pair[0]
        q = pair[1]
        alpha = dict()
        for letter in alphabet:
            outputp = output_function(transducer, letter, p)
            if type(outputp) == list:
                outputp = str(outputp[0])
            outputq = output_function(transducer, letter, q)
            if type(outputq) == list:
                outputq = str(outputq[0])
            outputp = alphabet.index(outputp)
            outputq = alphabet.index(outputq)
            alpha[outputq] = outputp
        alphadc = []
        for x in range(len(alphabet)):
            inanycycle = False
            for cycle in alphadc:
                if x in cycle:
                    inanycycle = True
            if not inanycycle:
                newcycle = [x]
                keepgoing = True
                startletter = x
                while keepgoing:
                    y = alpha[x]
                    if y == startletter:
                        keepgoing = False
                    else:
                        newcycle.append(y)
                        x = y
                min_index = newcycle.index(min(newcycle))
                newcycle = newcycle[min_index:] + newcycle[:min_index]
                alphadc.append(newcycle)
        keepgoing = True
        i = 0
        while keepgoing:
            B_i = synchronizing_sequence(B,i)
            B_iStateDict = synch_seq_identified_states(B,i)
            if B_iStateDict[p] != B_iStateDict[q]:
                allParallel = True
                for letter1 in alphabet:
                    for letter2 in alphabet:
                        InSameCycle = False
                        for cycle in alphadc:
                            if alphabet.index(letter1) in cycle and alphabet.index(letter2) in cycle:
                                InSameCycle = True
                        if InSameCycle:
                            endstate1 = transition_function(B_i, letter1, B_iStateDict[q])
                            endstate2 = transition_function(B_i, letter2, B_iStateDict[q])
                            if endstate1 != endstate2:
                                allParallel = False
                if allParallel:
                    keepgoing = False
                    i = i - 1
            i = i + 1
        vertices = get_state_list(B_i)
        V = dict()
        for vertex in vertices:
            V[vertex] = vertex
        edges = [(edge[0],edge[2],edge[1]) for edge in B_i]
        E = dict()
        for edge in edges:
            if edge[0] == B_iStateDict[q]:
                E[edge] = (edge[0],alphabet[alpha[alphabet.index(edge[1])]],edge[2])
            else:
                E[edge] = edge
        tau = (V,E)
        TorsionElement = transducer_from_graph_automorphism(B_i, tau)
        TorsionElementsList.append(TorsionElement)
        R = take_core(transducer_product(transducer, TorsionElement))
        transducer = combine_equivalent_states(R)
        states = get_state_list(transducer)
    OneStateInverse = invert(transducer)
    TorsionElementsList.append(OneStateInverse)
    TorsionElementsList = [rename_nodes_int(invert(i)) for i in TorsionElementsList]
    for i in range(len(TorsionElementsList)):
        for j in range(len(TorsionElementsList[i])):
            TorsionElementsList[i][j][3] = [TorsionElementsList[i][j][3]]
    TorsionElementsList = TorsionElementsList[::-1]
    return TorsionElementsList

def Hn_product(transducer1, transducer2):

    # Returns the product (in the group H_n) of two elements of H_n
    
    if not IsInHn(transducer1) or not IsInHn(transducer2):
        raise ValueError("Transducers must by in H_n")
    product = transducer_product(transducer1, transducer2)
    coreofprod = take_core(product)
    return combine_equivalent_states(coreofprod)

def variable_check(objectToBeNamed, objectDescriptor, parentFunction):
    
# objectToBeNamed:  any Python object
# objectDescriptor: a string of a couple words that describe what the object is,
#                   e.g. 'transducer', 'output word'
# parentFunction:   the function under which variable_check is called

    keepLooping = True
    exitCondition = False
    while keepLooping:
        print()
        print("What variable name do you wish the " + objectDescriptor + " to have?")
        print()
        variableName = str(input("---> "))
        if variableName == "back":
            keepLooping = False
            parentFunction()
        elif variableName == "main menu":
            keepLooping = False
            main_menu()
        elif variableName == "exit":
            keepLooping = False
        else:
            if variableName in globals():
                print()
                print("'" + variableName + "' is currently assigned to a variable.")
                print("Are you sure you wish to overwrite this?")
                print()
                print("(type 'yes' or 'no')")
                print()
                auxVar3 = str(input("---> "))
                while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                    print()
                    print("Type 'yes' or 'no'")
                    print()
                    auxVar3 = str(input("---> "))
                if auxVar3 == "main menu":
                    keepLooping = False
                    main_menu()
                if auxVar3 == "exit":
                    keepLooping = False
                if auxVar3 == "yes":
                    keepLooping = False
                    exitCondition = True
                    globals()[variableName] = objectToBeNamed
                    print("The " + objectDescriptor + " has been saved as the variable '" + variableName + "'.")
            else:
                keepLooping = False
                exitCondition = True
                print()
                globals()[variableName] = objectToBeNamed
                print("The " + objectDescriptor + " has been saved as the variable '" + variableName + "'.")
            if exitCondition:
                parentFunction()
                
def main_menu():
    print()
    print("     #######  #####  ###")
    print("        #    #     #  #")
    print("        #    #        #")
    print("        #    #  ####  #")
    print("        #    #     #  #")
    print("        #    #     #  #")
    print("        #     #####  ###")
    print()
    print()
    print("Welcome to the Transducer Groups Interface. This module has many")
    print("functions, and can visualise automata and transducers, as well as")
    print("save them to your machine's memory in your preferred format.")
    print()
    print("At any time, type 'exit' to exit the interface and return to the")
    print("Python command line. If you want to re-enter the interface after this,")
    print("type 'main_menu()' into the command line.")
    print()
    print("Please choose from one of the following options below, by entering")
    print("the number of the option that you desire. You can type 'back' at any")
    print("time to return to the last menu you accessed, or 'main menu' to return here.")
    print()
    print("1) Create / save / load an automaton or transducer")
    print("2) Functions")
    print("3) Settings")
    print()
    auxVar = str(input("---> "))
    if auxVar == "1":
        transducer_interface()
    elif auxVar == "2":
        functions_interface()
    elif auxVar == "3":
        settings_interface()
    elif auxVar != "exit":
        main_menu()

def transducer_interface():
    print()
    print("Here you can create an automaton or transducer from scratch,")
    print("save a transducer to your specified directory, or load a transducer")
    print("from your directory into this Python session.")
    print()
    print("1) Create a new automaton or transducer")
    print("2) Endow an automaton with outputs")
    print("3) Generate automaton by forgetting the outputs of a transducer")
    print("4) Save an automaton or transducer to the directory")
    print("5) Load an automaton or transducer from the directory")
    print()
    auxVar = str(input("---> "))
    print()
    if auxVar == "1":
        create_transducer_interface()
    elif auxVar == "2":
        give_outputs_interface()
    elif auxVar == "3":
        forget_outputs_interface()
    elif auxVar == "4":
        save_transducer_interface()
    elif auxVar == "5":
        load_transducer_interface()
    elif auxVar == "back" or auxVar == "main menu":
        main_menu()
    elif auxVar != "exit":
        transducer_interface()

def create_transducer_interface():
    edgeList = []
    print()
    print("Would you like to create an automaton or transducer?")
    print()
    print("1) Automaton")
    print("2) Transducer")
    print()
    auxVar = str(input("---> "))
    if auxVar == "1":
        print()
        print("What is the size of the alphabet that the automaton is over?")
        print("Note: the alphabet will be saved as 0,1,2,...")
        auxVar = str(input("---> "))
        if auxVar == "back":
            create_transducer_interface()
        elif auxVar == "main menu":
            main_menu()
        elif auxVar == "exit":
            pass
        elif auxVar != "exit":
            auxVar = int(auxVar)
            alphabet = [str(i) for i in range(auxVar)]
            print()
            print("How many states does the automaton have?")
            print("Note: the states will be labelled 0,1,2,...")
            auxVar = str(input("---> "))
            if auxVar == "back":
                create_transducer_interface()
            elif auxVar == "main menu":
                main_menu()
            elif auxVar == "exit":
                pass
            elif auxVar != "exit":
                auxVar = int(auxVar)
                print()
                nodeList = [str(i) for i in range(auxVar)]
                num = len(nodeList)*len(alphabet)
                i = 0
                j = 0
                boolean = True
                final = False
                while boolean:
                    if not final:
                        print()
                        print("Which state does the automaton go to from state " + str(i))
                        print("after reading letter " + str(j) + "?")
                        auxVar = str(input("---> "))
                        flag = "good"
                        if auxVar == "main menu":
                            flag = "mm"
                        elif auxVar == "exit":
                            flag = "ex"
                        elif auxVar == "back":
                            if j == 0:
                                if i == 0:
                                    flag = "back"
                                else:
                                    j = len(alphabet)-1
                                    i = i - 1
                            else:
                                j = j - 1
                            index = i*len(alphabet) + j
                            if flag != "back":
                                del edgeList[index]
                        else:
                            if (i+1)*(j+1) == num:
                                final = True
                            print()
                            edgeList.append([str(i),auxVar,str(j)])
                            if j == len(alphabet) - 1:
                                i = i + 1
                                j = 0
                            else:
                                j = j + 1
                        if flag != "good":
                            boolean = False
                    if final:
                        keepLooping = True
                        exitCondition = False
                        while keepLooping:
                            print()
                            print("What variable name do you wish the automaton to have?")
                            print()
                            variableName = str(input("---> "))
                            if variableName == "back":
                                keepLooping = False
                                final = False
                                del edgeList[-1]
                                i = i - 1
                                j = len(alphabet) - 1
                            else:
                                boolean = False
                            if variableName == "main menu":
                                keepLooping = False
                                main_menu()
                            elif variableName == "exit":
                                keepLooping = False
                            elif variableName != "back":
                                if variableName in globals():
                                    print()
                                    print("'" + variableName + "' is currently assigned to a variable.")
                                    print("Are you sure you wish to overwrite this?")
                                    print()
                                    print("(type 'yes' or 'no')")
                                    print()
                                    auxVar3 = str(input("---> "))
                                    while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                                        print()
                                        print("Type 'yes' or 'no'")
                                        print()
                                        auxVar3 = str(input("---> "))
                                    if auxVar3 == "back":
                                        boolean = True
                                    if auxVar3 == "main menu":
                                        keepLooping = False
                                        main_menu()
                                    if auxVar3 == "exit":
                                        keepLooping = False
                                    if auxVar3 == "yes":
                                        keepLooping = False
                                        exitCondition = True
                                        globals()[variableName] = edgeList
                                        print("The automaton has been saved as the variable '" + variableName + "'.")
                                else:
                                    keepLooping = False
                                    exitCondition = True
                                    print()
                                    globals()[variableName] = edgeList
                                    print("The automaton has been saved as the variable '" + variableName + "'.")
                                if exitCondition:
                                    create_transducer_interface()
                if flag == "mm":
                    main_menu()
                elif flag == "ex":
                    pass
                elif flag == "back":
                    create_transducer_interface()
    elif auxVar == "2":
        print()
        print("What is the size of the alphabet that the transducer is over?")
        print("Note: the alphabet will be saved as 0,1,2,...")
        auxVar = str(input("---> "))
        if auxVar == "back":
            create_transducer_interface()
        elif auxVar == "main menu":
            main_menu()
        elif auxVar == "exit":
            pass
        elif auxVar != "exit":
            auxVar = int(auxVar)
            alphabet = [str(i) for i in range(auxVar)]
            print()
            print("How many states does the transducer have?")
            print("Note: the states will be labelled 0,1,2,...")
            auxVar = str(input("---> "))
            if auxVar == "back":
                create_transducer_interface()
            elif auxVar == "main menu":
                main_menu()
            elif auxVar == "exit":
                pass
            elif auxVar != "exit":
                auxVar = int(auxVar)
                print()
                nodeList = [str(i) for i in range(auxVar)]
                num = len(nodeList)*len(alphabet)
                i = 0
                j = 0
                boolean = True
                final = False
                while boolean:
                    if not final:
                        print()
                        print("Which state does the transducer go to from state " + str(i))
                        print("after reading letter " + str(j) + ", and what is the output?")
                        print()
                        print("Separate the state and output with a semi-colon, and give the output as a list.")
                        print("If output is one-letter, it need not be given as a list. For example,")
                        print("3; [1, 3, 2] or 2  ;[0   ,0] or 1;1 are all acceptable")
                        auxVar = str(input("---> "))
                        flag = "good"
                        if auxVar == "main menu":
                            flag = "mm"
                        elif auxVar == "exit":
                            flag = "ex"
                        elif auxVar == "back":
                            if j == 0:
                                if i == 0:
                                    flag = "back"
                                else:
                                    j = len(alphabet)-1
                                    i = i - 1
                            else:
                                j = j - 1
                            index = i*len(alphabet) + j
                            if flag != "back":
                                del edgeList[index]
                        else:
                            if (i+1)*(j+1) == num:
                                final = True
                            print()
                            auxVar1, auxVar2 = auxVar.split(";")
                            auxVar1 = auxVar1.split()[0]
                            auxVar1 = str(auxVar1)
                            auxVar2 = str(auxVar2)
                            auxVar2 = ast.literal_eval(auxVar2)
                            if type(auxVar2) != list:
                                auxVar2 = [auxVar2]
                            auxVar2 = [str(i) for i in auxVar2]
                            edgeList.append([str(i), auxVar1, str(j), auxVar2])
                            if j == len(alphabet) - 1:
                                i = i + 1
                                j = 0
                            else:
                                j = j + 1
                        if flag != "good":
                            boolean = False
                    if final:
                        keepLooping = True
                        exitCondition = False
                        while keepLooping:
                            print()
                            print("What variable name do you wish the transducer to have?")
                            print()
                            variableName = str(input("---> "))
                            if variableName == "back":
                                keepLooping = False
                                final = False
                                del edgeList[-1]
                                i = i - 1
                                j = len(alphabet) - 1
                            else:
                                boolean = False
                            if variableName == "main menu":
                                keepLooping = False
                                main_menu()
                            elif variableName == "exit":
                                keepLooping = False
                            elif variableName != "back":
                                if variableName in globals():
                                    print()
                                    print("'" + variableName + "' is currently assigned to a variable.")
                                    print("Are you sure you wish to overwrite this?")
                                    print()
                                    print("(type 'yes' or 'no')")
                                    print()
                                    auxVar3 = str(input("---> "))
                                    while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                                        print()
                                        print("Type 'yes' or 'no'")
                                        print()
                                        auxVar3 = str(input("---> "))
                                    if auxVar3 == "back":
                                        boolean = True
                                    if auxVar3 == "main menu":
                                        keepLooping = False
                                        main_menu()
                                    if auxVar3 == "exit":
                                        keepLooping = False
                                    if auxVar3 == "yes":
                                        keepLooping = False
                                        exitCondition = True
                                        globals()[variableName] = edgeList
                                        print("The transducer has been saved as the variable '" + variableName + "'.")
                                else:
                                    keepLooping = False
                                    exitCondition = True
                                    print()
                                    globals()[variableName] = edgeList
                                    print("The transducer has been saved as the variable '" + variableName + "'.")
                                if exitCondition:
                                    create_transducer_interface()
                if flag == "mm":
                    main_menu()
                elif flag == "ex":
                    pass
                elif flag == "back":
                    create_transducer_interface()
    elif auxVar == "back":
        transducer_interface()
    elif auxVar == "main menu":
        main_menu()
    elif auxVar != "exit":
        transducer_interface()

def give_outputs_interface():
    print()
    print("Here you can generate a transducer from an automaton, by specifying")
    print("the outputs for each transition.")
    print()
    print("What is the variable name of the automaton?")
    print()
    variableName1 = str(input("---> "))
    if variableName1 == "back":
        transducer_interface()
    elif variableName1 == "main menu":
        main_menu()
    elif variableName1 != "exit":
        if variableName1 in globals():
            automaton = globals()[variableName1]
            newTransducer = []
            i = 0
            num = len(automaton)
            boolean = True
            final = False
            while boolean:
                if not final:
                    print()
                    print("What is the output from state " + automaton[i][0] + " when reading letter " + automaton[i][1] + "?")
                    auxVar2 = str(input("---> "))
                    flag = "good"
                    if auxVar2 == "main menu":
                        flag = "mm"
                    elif auxVar2 == "exit":
                        flag = "ex"
                    elif auxVar2 == "back":
                        if i == 0:
                            flag = "back"
                        else:
                            i = i - 1
                            del newTransducer[i]
                    else:
                        if i + 1 == num:
                            final = True
                        newTransducer.append([automaton[i][0], automaton[i][1], automaton[i][2], auxVar2])
                        i = i + 1
                    if flag != "good":
                        boolean = False
                if final:
                    keepLooping = True
                    exitCondition = False
                    while keepLooping:
                        print()
                        print("What variable name do you wish the transducer to have?")
                        print()
                        variableName = str(input("---> "))
                        if variableName == "back":
                            keepLooping = False
                            final = False
                            del newTransducer[-1]
                            i = i - 1
                        else:
                            boolean = False
                        if variableName == "main menu":
                            keepLooping = False
                            main_menu()
                        elif variableName == "exit":
                            keepLooping = False
                        elif variableName != "back":
                            if variableName in globals():
                                print()
                                print("'" + variableName + "' is currently assigned to a variable.")
                                print("Are you sure you wish to overwrite this?")
                                print()
                                print("(type 'yes' or 'no')")
                                print()
                                auxVar3 = str(input("---> "))
                                while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                                    print()
                                    print("Type 'yes' or 'no'")
                                    print()
                                    auxVar3 = str(input("---> "))
                                if auxVar3 == "back":
                                    boolean = True
                                if auxVar3 == "main menu":
                                    keepLooping = False
                                    main_menu()
                                if auxVar3 == "exit":
                                    keepLooping = False
                                if auxVar3 == "yes":
                                    keepLooping = False
                                    exitCondition = True
                                    globals()[variableName] = newTransducer
                                    print("The transducer has been saved as the variable '" + variableName + "'.")
                            else:
                                keepLooping = False
                                exitCondition = True
                                print()
                                globals()[variableName] = newTransducer
                                print("The transducer has been saved as the variable '" + variableName + "'.")
                            if exitCondition:
                                give_outputs_interface()
            if flag == "mm":
                main_menu()
            elif flag == "back":
                give_outputs_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                give_outputs_interface()

def forget_outputs_interface():
    print()
    print("Here you can generate an automaton from a transducer,")
    print("by discarding the outputs.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName1 = str(input("---> "))
    if variableName1 == "back":
        transducer_interface()
    elif variableName1 == "main menu":
        main_menu()
    elif variableName1 != "exit":
        if variableName1 in globals():
            transducer = globals()[variableName1]
            newAutomaton = [[edge[0], edge[1], edge[2]] for edge in transducer]
            variable_check(newAutomaton, "automaton", forget_outputs_interface)
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                forget_outputs_interface()

def save_transducer_interface():
    print()
    print("Here you can take an automaton or transducer that is currently")
    print("stored as a variable in this Python session, and save it as a")
    print(".txt file to the directory. The transducer can be in the .txt file")
    print("in a variety of formats. For more information on this, type 'formats'.")
    print()
    print("What is the name of the variable under which your automaton / transducer")
    print("is currently saved?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        transducer_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName == "exit":
        pass
    elif variableName == "formats":
        formats_interface(save_transducer_interface)
    elif variableName != "exit":
        if variableName in globals():
            auxVar1 = globals()[variableName]
            if type(auxVar1) == list:
                if len(auxVar1[0]) == 4:
                    which = "transducer"
                else:
                    which = "automaton"
                print()
                print("What format would you like to save '" + str(variableName) + "' as?")
                print()
                print("1) edgeList")
                print("2) AAA")
                print("3) DOT")
                print()
                auxVar3 = str(input("---> "))
                if auxVar3 == "back":
                    save_transducer_interface()
                elif auxVar3 == "main menu":
                    main_menu()
                elif auxVar3 == "exit":
                    pass
                elif auxVar3 in ["1", "2", "3"]:
                    if auxVar3 == "1":
                        savingFunction = edgeList_to_txt
                    elif auxVar3 == "2":
                        auxVar1 = edgeList_to_aaa(auxVar1)
                        savingFunction = aaa_to_txt
                    elif auxVar3 == "3":
                        auxVar1 = edgeList_to_graph(auxVar1)
                        savingFunction = graph_to_txt
                    keepLooping = True
                    exitCondition = False
                    while keepLooping:
                        print()
                        print("What would you like the name of the file to be?")
                        print()
                        auxVar2 = str(input("---> "))
                        if auxVar2 == "back":
                            keepLooping = False
                            save_transducer_interface()
                        elif auxVar2 == "main menu":
                            keepLooping = False
                            main_menu()
                        elif auxVar2 == "exit":
                            keepLooping = False
                        else:
                            try:
                                with open(auxVar2 + ".txt", "r") as data:
                                    pass
                                print()
                                print("There already exists a file called '" + auxVar2 + "' in the directory.")
                                print("Are you sure you wish to overwrite this?")
                                print()
                                print("(type 'yes' or 'no')")
                                print()
                                auxVar4 = str(input("---> "))
                                while auxVar4 != 'yes' and auxVar4 != 'no' and auxVar4 not in ["main menu", "back", "exit"]:
                                    print()
                                    print("Type 'yes' or 'no'")
                                    print()
                                    auxVar4 = str(input("---> "))
                                if auxVar4 == "main menu":
                                    keepLooping = False
                                    main_menu()
                                if auxVar4 == "exit":
                                    keepLooping = False
                                if auxVar4 == "yes":
                                    keepLooping = False
                                    exitCondition = True
                                    savingFunction(auxVar1, auxVar2)
                                    print("The " + which + " has been saved as '" + auxVar2 + ".txt'.")
                            except FileNotFoundError:
                                keepLooping = False
                                exitCondition = True
                                savingFunction(auxVar1, auxVar2)
                                print("The " + which + " has been saved as '" + auxVar2 + ".txt'.")
                            if exitCondition:
                                save_transducer_interface()
        else:
            print("No such variable")
            print()
            print("Type anything to return to the previous menu")
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar == "exit":
                pass
            elif auxVar != "exit":
                save_transducer_interface()

def load_transducer_interface():
    print()
    print("Here you can take an automaton or transducer that is currently")
    print("saved as a .txt file in the directory, and save it as a variable")
    print("in this Python session. The transducer can be in the .txt file")
    print("in a variety of formats. For more information on this, type 'formats'.")
    print()
    print("What is the name of the .txt file in the directory that you wish")
    print("to load? e.g., if the file is called example.txt, type 'example'")
    print()
    auxVar = str(input("---> "))
    if auxVar == "back":
        transducer_interface()
    elif auxVar == "main menu":
        main_menu()
    elif auxVar == "exit":
        pass
    elif auxVar == "formats":
        formats_interface(load_transducer_interface)
    elif auxVar != "exit":
        try:
            transducer = txt_to_edgeList(auxVar)
            if len(transducer[0]) == 4:
                variable_check(transducer, "transducer", load_transducer_interface)
            else:
                variable_check(transducer, "automaton", load_transducer_interface)
        except FileNotFoundError:
            print()
            print("File not found. Please try again.")
            load_transducer_interface()

def formats_interface(parentFunction):
    print()
    print("This module can save / load .txt files in the following formats:")
    print("edgeList format, the AAA format, DOT format.")
    print()
    print("1) The edgeList format is a format designed for readability.")
    print("   Each line of the .txt file represents an edge of the automaton /")
    print("   transducer, with a semi-colon and a space seperating each piece")
    print("   of information, e.g. one line of the .txt file might read '0; 2; 1; [1,0]',")
    print("   which means the transducer goes from state 0 to state 2 when reading")
    print("   letter 1, and the output is [1,0]. Synchronous transducers may have their")
    print("   outputs written as a letter, e.g. one line could be '0; 2; 1; 1'.")
    print("   Automata are precisely as you'd expect, with outputs dropped, e.g. '0; 2; 1'.")
    print()
    print("2) The AAA format is the format used in the AAA")
    print("   ('asynchronous automata algoriths') GAP package by")
    print("   Collin Bleak, Fernando Flores Brito, Luke Elliott and Feyishayo Olukoya.")
    print("   In that package, transducers are treated as 4-tuples (m, n, P, L).")
    print("   m is the size of the input alphabet, n is the size of the output alphabet,")
    print("   P is a list of lists describing the transitions of the transducer, and")
    print("   L is a list of list of lists describing the outputs. For example,")
    print()
    print("   (2, 2, [[2, 3], [2, 3], [2, 3]], [[[], []], [[0], [0]], [[1], [1]]])")
    print()
    print("   represents a transducer on a 2-letter alphabet. the element of index i")
    print("   in the list P gives the transitions going from state i, e.g. in this")
    print("   transducer, there are 3 states, and reading 0 always transitions to")
    print("   state 2, while reading 1 always transitions to state 3. Similarly,")
    print("   the element of index i in the list L gives the outputs for each letter,")
    print("   with the outputs themselves written as a list of each letter in the output")
    print("   word, e.g. in this transducer, state 1 always gives empty outputs, state 2")
    print("   always gives output 0, and state 3 always gives output 1.")
    print()
    print("   A .txt file of a transducer can be loaded in if it consists of one line of")
    print("   of text, which is the above 4-tuple, which can be a list instead of a tuple,")
    print("   and can be preceded by the word 'Transducer', as it would be when setting")
    print("   it as a variable in GAP. It can be saved into a .txt file as the tuple.")
    print()        
    print("3) DOT is a graph description language, with a syntax designed for")
    print("   encoding graphs. For more information, see:")
    print()
    print("   https://www.graphviz.org/doc/info/lang.html")
    print()
    print("   This program requires a .txt or .dot file in DOT format to describe")
    print("   a labeled directed graph that represents an automaton or transducer.")
    print("   For transducers, the label must be of the form 'a|b', where 'a' is the")
    print("   input letter and 'b' is the output word (as a list, or as a letter if the")
    print("   transducer is synchronous) with no spaces between 'a', '|' (which, for")
    print("   clarity, is the vertical bar symbol with unicode code point 'U+007C'), and 'b'.")
    print("   For automata, the label is simply the input letter 'a'.")
    print()
    print("Type anything to return to the previous menu.")
    auxVar = str(input("---> "))
    if auxVar == "exit":
        pass
    elif auxVar == "main menu":
        main_menu()
    elif auxVar != "exit":
        parentFunction()

def functions_interface():
    print()
    print("In order to use the following functions, you must have already")
    print("saved your automata / transducers as variables into this Python session.")
    print()
    print("Choose from the following options.")
    print()
    print("1)  Generate an image of an automaton or transducer")
    print("2)  Give the transitions of an automaton, or the transitions / outputs")
    print("    of a transducer")
    print("3)  Take the product of two transducers")
    print("4)  Find the words that force a given state of a")
    print("    strongly synchronizing automaton / transducer")
    print("5)  Find the level k dual of a transducer")
    print("6)  Generate the de Bruijn automaton G(n,m)")
    print("7)  Construct an automorphism of the underlying graph of an automaton")
    print("8)  Check if a map from the underlying graph of an automaton")
    print("    to itself is a graph automorphism")
    print("9)  Construct a transducer from an automaton and an automorphism")
    print("    of its underlying graph")
    print("10) Partition a transducer's states into omega-equivalence classes")
    print("11) Check if a transducer is weakly minimal")
    print("12) Identify omega-equivalent states of a transducer")
    print("13) Find a term in the synchronizing sequence of an automaton")
    print("    (or of the underlying automaton of a transducer)")
    print("14) Check if an automaton or transducer is strongly synchronizing")
    print("15) Find the minimal synchronzing level of an automaton or transducer")
    print("16) Check if a strongly synchronizing automaton or transducer is core")
    print("17) Generate the core of a strongly synchronizing automaton or transducer")
    print("18) Check if a transducer is synchronous")
    print("19) Check if a synchronous transducer is invertible")
    print("20) Invert a synchronous, invertible transducer")
    print("21) Check if a synchronous, invertible transducer is bi-synchronizing")
    print("22) Check if a transducer is in the group H_n")
    print("23) Decompose an element of H_n into torsion elements")
    print("24) Take the product in H_n of two elements")
    print()
    auxVar = str(input("---> "))
    print()
    if auxVar == "back" or auxVar == "main menu":
        main_menu()
    elif auxVar == "exit":
        pass
    elif auxVar == "1":
        generate_image_interface()
    elif auxVar == "2":
        transitions_outputs_interface()
    elif auxVar == "3":
        product_interface()
    elif auxVar == "4":
        forcing_words_interface()
    elif auxVar == "5":
        dual_interface()
    elif auxVar == "6":
        de_bruijn_interface()
    elif auxVar == "7":
        create_graph_aut_interface()
    elif auxVar == "8":
        check_graph_aut_interface()
    elif auxVar == "9":
        H_A_phi_interface()
    elif auxVar == "10":
        omega_partition_interface()
    elif auxVar == "11":
        check_weakly_min_interface()
    elif auxVar == "12":
        equiv_states_interface()
    elif auxVar == "13":
        synch_seq_interface()
    elif auxVar == "14":
        check_strong_synch_interface()
    elif auxVar == "15":
        synch_level_interface()
    elif auxVar == "16":
        check_core_interface()
    elif auxVar == "17":
        core_interface()
    elif auxVar == "18":
        check_synchronous_interface()
    elif auxVar == "19":
        check_automata_inv_interface()
    elif auxVar == "20":
        invert_synchronous_interface()
    elif auxVar == "21":
        check_bisynch_interface()
    elif auxVar == "22":
        check_in_Hn_interface()
    elif auxVar == "23":
        decomp_interface()
    elif auxVar == "24":
        Hn_product_interface()
    elif auxVar != "exit":
        functions_interface()

def generate_image_interface():
    print()
    print("This function generates a .png file in the directory, showing an")
    print("image of the automaton or transducer that you give it.")
    print()
    print("Enter the name of the variable under which the automaton / transducer")
    print("is currently saved.")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName == "exit":
        pass
    elif variableName != "exit":
        if variableName in globals():
            auxVar1 = globals()[variableName]
            print()
            print("What would you like the name of the .png file to be?")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "back":
                generate_image_interface()
            elif auxVar2 == "main menu":
                main_menu()
            elif auxVar2 == "exit":
                pass
            elif auxVar2 != "exit":
                nodeList = get_state_list(auxVar1)
                if len(nodeList) > 100 or len(auxVar1) > 300:
                    print()
                    print("Your automaton / transducer has " + str(len(nodeList)) + " states and " + str(len(auxVar1)) + " transitions.")
                    print("Generating images of large automata / transducers can be slow, memory-intensive,")
                    print("and buggy. For example, on my machine (a decent laptop), it took about")
                    print("a minute to generate an image of an automaton with 100 states and 300 transitions.")
                    print("When I tried to generate an image of one with 256 states and 1024 transitions,")
                    print("this program stopped working. Are you sure you wish to proceed?")
                    print()
                    print("Type 'yes' or 'no'.")
                    print()
                    auxVar3 = str(input("---> "))
                    while auxVar3 != "yes" and auxVar3 != "no" and auxVar3 not in ["main menu", "back", "exit"]:
                        print("Type 'yes' or 'no'.")
                        print()
                        auxVar3 = str(input("---> "))
                    if auxVar3 == "no":
                        generate_image_interface()
                    elif auxVar3 == "yes":
                        draw_graph(auxVar1, auxVar2)
                        print("The image has been generated as '" + auxVar2 + ".png'.")
                        generate_image_interface()
                    elif auxVar3 == "main menu":
                        main_menu()
                    elif auxVar3 == "back":
                        generate_image_interface()
                else:
                    draw_graph(auxVar1, auxVar2)
                    print("The image has been generated as '" + auxVar2 + ".png'.")
                    generate_image_interface()
        else:
            print("No such variable")
            print()
            print("Type anything to return to the previous menu")
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                generate_image_interface()

def transitions_outputs_interface():
    print()
    print("This function can tell you what the final state and output is, for")
    print("a given automaton / transducer and finite input word.")
    print()
    print("Enter the name of the variable under which the automaton / transducer")
    print("is currently saved.")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName == "exit":
        pass
    elif variableName != "exit":
        if variableName in globals():
            auxVar1 = globals()[variableName]
            print()
            print("What is the start state?")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "main menu":
                main_menu()
            elif auxVar2 == "back":
                transitions_outputs_interface()
            elif auxVar2 != "exit":
                print()
                print("What is the input word?")
                print("Type the word as a list consisting of its letters. This is not")
                print("necessary if it is a one-letter word.")
                print()
                auxVar3 = str(input("---> "))
                if auxVar3 == "main menu":
                    main_menu()
                elif auxVar3 == "back":
                    transitions_outputs_interface()
                elif auxVar3 != "exit":
                    auxVar3 = ast.literal_eval(auxVar3)
                    auxVar3 = [str(i) for i in auxVar3]
                    print()
                    if len(auxVar1[0]) == 4:
                        print("The final state is: ", transition_function(auxVar1,auxVar3,auxVar2))
                        print("The output word is: ", output_function(auxVar1,auxVar3,auxVar2))
                        print()
                        print("Type anything to return to the previous menu")
                        print()
                        auxVar = str(input("---> "))
                        if auxVar == "main menu":
                            main_menu()
                        elif auxVar != "exit":
                            transitions_outputs_interface()
                    else:
                        print("The final state is: ", transition_function(auxVar1,auxVar3,auxVar2))
                        print()
                        print("Type anything to return to the previous menu")
                        print()
                        auxVar = str(input("---> "))
                        if auxVar == "main menu":
                            main_menu()
                        elif auxVar != "exit":
                            transitions_outputs_interface()
        else:
            print()
            print("No such variable. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                transitions_outputs_interface()

def product_interface():
    print()
    print("Here you can take two transducers that are saved as variables to this")
    print("Python session, and that are over the same alphabet, and generate")
    print("their transducer product.")
    print()
    print("We are calculating transducer1 * transducer2")
    print()
    print("What is the variable name belonging to transducer1?")
    print()
    auxVar1 = str(input("---> "))
    if auxVar1 == "back":
        functions_interface()
    elif auxVar1 == "main menu":
        main_menu()
    elif auxVar1 != "exit":
        if auxVar1 in globals():
            transducer1 = globals()[auxVar1]
            print()
            print("What is the variable name belonging to transducer2?")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "back":
                product_interface()
            elif auxVar2 == "main menu":
                main_menu()
            elif auxVar2 != "exit":
                if auxVar2 in globals():
                    transducer2 = globals()[auxVar2]
                    product = transducer_product(transducer1, transducer2)
                    variable_check(product, "product", product_interface)
                else:
                    print("No such variable found. Type anything to return to the previous menu.")
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        product_interface()
        else:
            print("No such variable found. Type anything to return to the previous menu.")
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                product_interface()

def forcing_words_interface():
    print()
    print("Given a strongly synchronizing automaton or transducer, this function")
    print("tells you all the words of a given length that force a given state.")
    print()
    print("What is the variable name of the strongly synchronizing automaton / transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            if not IsStronglySynchronizing(transducer):
                print()
                print("'" + str(variableName) + "' is not strongly synchronizing.")
                print("Type anything to return to the previous menu.")
                print("Type anything to return to the previous menu")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    forcing_words_interface()
            else:
                print()
                print("Which state do you wish to be forced by the words?")
                print()
                state = str(input("---> "))
                if state == "back":
                    forcing_words_interface()
                elif state == "main menu":
                    main_menu()
                elif state != "exit":
                    print()
                    print("What length do you wish the words to be?")
                    print("N.B. this function will only give valid output if the automaton / transducer")
                    print("is strongly synchronizing at the level of the number that you enter here.")
                    print()
                    length = str(input("---> "))
                    if length == "back":
                        forcing_words_interface()
                    elif length == "main menu":
                        main_menu()
                    elif length != "exit":
                        length = int(length)
                        forcingWords = forcing_words_for_state(transducer, state, length)
                        print()
                        print("The words of length " + str(length) + " that force state " + state + " are:")
                        print()
                        print(forcingWords)
                        print()
                        print("Type anything to return to the previous menu")
                        print()
                        auxVar = str(input("---> "))
                        if auxVar == "main menu":
                            main_menu()
                        elif auxVar != "exit":
                            forcing_words_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar1 = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                forcing_words_interface()

def dual_interface():
    print()
    print("Here you can generate the level k dual of a synchronous transducer.")
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            if not IsSynchronous(transducer):
                print("Transducer is not synchronous.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    dual_interface()
            else:
                print()
                print("Which level dual do you wish to generate?")
                print()
                auxVar = str(input("---> "))
                if auxVar == "back":
                    dual_interface()
                elif auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    auxVar = int(auxVar)
                    dual = dual_level_k(transducer, auxVar)
                    variable_check(dual, "level " + str(auxVar) + " dual", dual_interface)
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                dual_interface()

def de_bruijn_interface():
    print()
    print("Here you can generate a de Bruijn automaton, and save it as a variable.")
    print()
    print("What size alphabet do you wish the de Bruijn automaton to be over?")
    print("(the 'n' value)")
    print()
    auxVar1 = str(input("---> "))
    if auxVar1 == "back":
        functions_interface()
    elif auxVar1 == "main menu":
        main_menu()
    elif auxVar1 != "exit":
        boolean1 = True
        boolean2 = True
        while boolean1:
            try:
                auxVar1 = int(auxVar1)
                boolean1 = False
            except (TypeError, ValueError):
                print()
                print("'n' must be an integer. What size alphabet do you wish")
                print("the de Bruijn automaton to have?")
                print()
                auxVar1 = str(input("---> "))
                if auxVar1 == "main menu":
                    boolean1 = False
                    boolean2 = False
                    main_menu()
                elif auxVar1 == "back":
                    boolean1 = False
                    boolean2 = False
                    de_bruijn_interface()
                elif auxVar1 == "exit":
                    boolean1 = False
                    boolean2 = False
        if boolean2:
            print()
            print("What synchronizing level do you wish the de Bruijn automaton to have?")
            print("(the 'm' value, so that G(n,m) is generated, with state set X_n^m)")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "back":
                de_bruijn_interface()
            elif auxVar2 == "main menu":
                main_menu()
            elif auxVar2 != "exit":
                boolean3 = True
                boolean4 = True
                while boolean3:
                    try:
                        auxVar2 = int(auxVar2)
                        boolean3 = False
                    except (TypeError, ValueError):
                        print()
                        print("'m' must be an integer. What synchronizing level do you wish")
                        print("the de Bruijn automaton to have?")
                        print()
                        auxVar2 = str(input("---> "))
                        if auxVar2 == "main menu":
                            boolean3 = False
                            boolean4 = False
                            main_menu()
                        elif auxVar2 == "back":
                            boolean3 = False
                            boolean4 = False
                            de_bruijn_interface()
                        elif auxVar2 == "exit":
                            boolean3 = False
                            boolean4 = False
                if boolean4:
                    automaton = de_bruijn_automaton(auxVar1, auxVar2)
                    variable_check(automaton, "de Bruijn automaton G(" + str(auxVar1) + "," + str(auxVar2) + ")", de_bruijn_interface)

def create_graph_aut_interface():
    print()
    print("Here you can generate a Python object called a dictionary that")
    print("represents an automorphism of the underlying graph of an automaton")
    print("that is currently saved as a variable in this Python session.")
    print("This is the format required for working with graph automorphisms")
    print("within this software.")
    print()
    print("What is the variable name of the automaton?")
    print()
    variableName1 = str(input("---> "))
    if variableName1 == "back":
        functions_interface()
    elif variableName1 == "main menu":
        main_menu()
    elif variableName1 != "exit":
        if variableName1 in globals():
            automaton = globals()[variableName1]
            vertices = [str(i) for i in get_state_list(automaton)]
            vertexdict = dict()
            edges = [(edge[0], edge[2], edge[1]) for edge in automaton]
            edgedict = dict()
            print()
            print("Will all vertices be mapped to themselves?")
            print()
            auxVar1 = ""
            while auxVar1 not in ["yes", "no", "main menu", "back", "exit"]:
                print("Type 'yes' or 'no'.")
                auxVar1 = str(input("---> "))
            if auxVar1 == "main menu":
                main_menu()
            elif auxVar1 == "back":
                create_graph_aut_interface()
            elif auxVar1 == "yes" or auxVar1 == "no":
                if auxVar1 == "yes":
                    i = len(vertices)
                    for vertex in vertices:
                        vertexdict[vertex] = vertex
                else:
                    i = 0
                num = len(vertices) + len(edges)
                boolean = True
                final = False
                while boolean:
                    if not final:
                        if i < len(vertices):
                            print()
                            print("Which vertex does the vertex " + vertices[i] + " map to?")
                            print("Type the name of a state of the automaton.")
                            auxVar2 = str(input("---> "))
                            flag = "good"
                            if auxVar2 == "main menu":
                                flag = "mm"
                            elif auxVar2 == "exit":
                                flag = "ex"
                            elif auxVar2 == "back":
                                if i == 0:
                                    flag = "back"
                                else:
                                    i = i - 1
                                    del vertexdict[vertices[i]]
                            else:
                                if i + 1 == num:
                                    final = True
                                vertexdict[vertices[i]] = auxVar2
                                i = i + 1
                            if flag != "good":
                                boolean = False
                        else:
                            j = i - len(vertices)
                            print()
                            print("Which edge does the edge " + str(edges[j]) + " map to?")
                            print("Considering the edge as a 3-tuple (p,x,q), going from vertex p to vertex q")
                            print("with label x, input the edge by typing each of the 3 elements p, x, q")
                            print("in order, separated by semi-colons. For example, 1;2;1.")
                            auxVar2 = str(input("---> "))
                            flag = "good"
                            if auxVar2 == "main menu":
                                flag = "mm"
                            elif auxVar2 == "exit":
                                flag = "ex"
                            elif auxVar2 == "back":
                                i = i - 1
                                if j != 0:
                                    del edgedict[edges[j]]
                            else:
                                if i + 1 == num:
                                    final = True
                                p, x, q = auxVar2.split(";")
                                p = p.split()[0]
                                x = x.split()[0]
                                q = q.split()[0]
                                auxVar2 = (str(p), str(x), str(q))
                                edgedict[edges[j]] = auxVar2
                                i = i + 1
                            if flag != "good":
                                boolean = False
                    if final:
                        keepLooping = True
                        exitCondition = False
                        while keepLooping:
                            print()
                            print("What variable name do you wish the automorphism to have?")
                            print()
                            variableName = str(input("---> "))
                            if variableName == "back":
                                keepLooping = False
                                final = False
                                del edgedict[edges[-1]]
                                i = i - 1
                            else:
                                boolean = False
                            if variableName == "main menu":
                                keepLooping = False
                                main_menu()
                            elif variableName == "exit":
                                keepLooping = False
                            elif variableName != "back":
                                if variableName in globals():
                                    print()
                                    print("'" + variableName + "' is currently assigned to a variable.")
                                    print("Are you sure you wish to overwrite this?")
                                    print()
                                    print("(type 'yes' or 'no')")
                                    print()
                                    auxVar3 = str(input("---> "))
                                    while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                                        print()
                                        print("Type 'yes' or 'no'")
                                        print()
                                        auxVar3 = str(input("---> "))
                                    if auxVar3 == "back":
                                        boolean = True
                                    if auxVar3 == "main menu":
                                        keepLooping = False
                                        main_menu()
                                    if auxVar3 == "exit":
                                        keepLooping = False
                                    if auxVar3 == "yes":
                                        keepLooping = False
                                        exitCondition = True
                                        globals()[variableName] = (vertexdict, edgedict)
                                        if IsValidAutomorphismForAutomaton(automaton, (vertexdict, edgedict)):
                                            print("The automorphism has been saved as the variable '" + variableName + "'.")
                                        else:
                                            print("This is not a valid graph automorphism. However, the map has")
                                            print("still been saved as the variable '" + variableName + "'.")
                                else:
                                    keepLooping = False
                                    exitCondition = True
                                    print()
                                    globals()[variableName] = (vertexdict, edgedict)
                                    if IsValidAutomorphismForAutomaton(automaton, (vertexdict, edgedict)):
                                        print("The automorphism has been saved as the variable '" + variableName + "'.")
                                    else:
                                        print("This is not a valid graph automorphism. However, the map has")
                                        print("still been saved as the variable '" + variableName + "'.")
                                if exitCondition:
                                    create_graph_aut_interface()
                if flag == "mm":
                    main_menu()
                elif flag == "back":
                    create_graph_aut_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                create_graph_aut_interface()

def check_graph_aut_interface():
    print()
    print("This function allows you to confirm whether or not a candidate graph")
    print("automorphism of the underlying graph of an automaton is a valid automorphism.")
    print()
    print("This will require both the automaton and the potential automorphism")
    print("to be saved as variables in the Python session already. The automorphism")
    print("is required to have a very specific format. It must be a 2-tuple (V,E),")
    print("where V and E are dicts that represent mappings from either the vertex")
    print("set or the edge set to themselves. Vertices that are being mapped in")
    print("V must be states of the automaton. Edges that are being mapped in E")
    print("must be in the form of the 3-tuple (p,x,q), where p is the vertex the edge")
    print("is coming from, q is the vertex the edge is going to, and x is the label")
    print("of the edge. This can be generated by this interface, in the 'Functions' menu.")
    print()
    print("What is the variable name of the automaton?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            automaton = globals()[variableName]
            print()
            print("What is the variable name of the potential automorphism?")
            print()
            variableName2 = str(input("---> "))
            if variableName2 == "back":
                check_graph_aut_interface()
            elif variableName2 == "main menu":
                main_menu()
            elif variableName2 != "exit":
                if variableName2 in globals():
                    automorph = globals()[variableName2]
                    print()
                    if IsValidAutomorphismForAutomaton(automaton, automorph):
                        print("This is a valid automorphism of the automaton's underlying graph.")
                    else:
                        print("This is not a valid automorphism of the automaton's underlying graph.")
                    print("Type anything to return to the previous menu.")
                    print()
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        check_graph_aut_interface()
                else:
                    print()
                    print("No such variable was found. Type anything to return to the previous menu.")
                    print()
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        check_graph_aut_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_graph_aut_interface()

def H_A_phi_interface():
    print()
    print("Here you can generate the transducer H(A, phi) from an automaton A and a valid")
    print("graph automorphism phi of the underlying graph of A.")
    print()
    print("The automorphism must be inputted as a 2-tuple (V, E), where V is a dict")
    print("that permutes the vertices (which are the states of the automaton), and")
    print("E is a dict that permutes the edges (which are 3-tuples (p,x,q), that go")
    print("from vertex p to vertex q, with label x).")
    print()
    print("What is the variable name belonging to the automaton?")
    print()
    auxVar1 = str(input("---> "))
    if auxVar1 == "back":
        functions_interface()
    elif auxVar1 == "main menu":
        main_menu()
    elif auxVar1 != "exit":
        if auxVar1 in globals():
            automaton = globals()[auxVar1]
            print()
            print("What is the variable name belonging to the graph automorphism?")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "back":
                H_A_phi_interface()
            elif auxVar2 == "main menu":
                main_menu()
            elif auxVar2 != "exit":
                if auxVar2 in globals():
                    automorphism = globals()[auxVar2]
                    if not IsValidAutomorphismForAutomaton(automaton, automorphism):
                        print()
                        print("Not a valid automorphism. Type anything to return to the previous menu.")
                        auxVar = str(input("---> "))
                        if auxVar == "main menu":
                            main_menu()
                        elif auxVar != "exit":
                            H_A_phi_interface()
                    else:
                        transd = transducer_from_graph_automorphism(automaton, automorphism)
                        variable_check(transd, "transducer", H_A_phi_interface)
                else:
                    print("No such variable found. Type anything to return to the previous menu.")
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        H_A_phi_interface()
        else:
            print("No such variable found. Type anything to return to the previous menu.")
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                H_A_phi_interface()

def omega_partition_interface():
    print()
    print("This function gives a partition of the states of a transducer into their")
    print("omega-equivalence classes.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            print()
            print("The below is the partition of the states into omega-equivalence classes:")
            print()
            print(omega_equivalence_partition(transducer))
            print()
            print("Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                omega_partition_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                omega_partition_interface()

def check_weakly_min_interface():
    print()
    print("Here you can check whether a transducer is weakly minimal.")
    print("Enter the variable name of the transducer.")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            print()
            transducer = globals()[variableName]
            auxVar = omega_equivalence_partition(transducer)
            badList = []
            boolean = True
            for subset in auxVar:
                if len(subset) != 1:
                    boolean = False
                    badList.append(subset)
            print()
            if boolean:
                print("The transducer '" + variableName + "' is weakly minimal.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_weakly_min_interface()
            else:
                print("The transducer '" + variableName + "' is not weakly minimal.")
                print()
                print("Would you like to see the states that are omega-equivalent?")
                print("Type 'yes' or 'no'.")
                print()
                auxVar = str(input("---> "))
                while auxVar != "yes" and auxVar != "no" and auxVar not in ["main menu", "back", "exit"]:
                    print()
                    print("Type 'yes' or 'no'.")
                    print()
                    auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                if auxVar == "back":
                    check_weakly_min_interface()
                if auxVar == "yes":
                    print()
                    for subset in badList:
                        string = "The states "
                        for element in subset:
                            if subset.index(element) == len(subset) - 1:
                                string = string + "'" + str(element) + "'" + " are omega-equivalent."
                            else:
                                string = string + "'" + str(element) + "'" + " and "
                        print(string)
                    print()
                    print("Type anything to return to the previous menu.")
                    print()
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        check_weakly_min_interface()
                if auxVar == "no":
                    print()
                    print("Type anything to return to the previous menu.")
                    print()
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        check_weakly_min_interface()
        else:
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_weakly_min_interface()

def equiv_states_interface():
    print()
    print("Here you can reduce a transducer to its weakly minimal representative,")
    print("by identifying all the states that are omega-equivalent to each other.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            transducer = combine_equivalent_states(transducer)
            variable_check(transducer, "resulting weakly minimal transducer", equiv_states_interface)
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                equiv_states_interface()

def synch_seq_interface():
    print()
    print("Here you can find the automata that are terms in the synchronizing")
    print("sequence of an automaton (or of the underlying automaton of a")
    print("transducer).")
    print()
    print("What is the variable name of the automaton / transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            automaton = globals()[variableName]
            if len(automaton[0]) == 4:
                automaton = transducer_to_automaton(automaton)
            print()
            print("Which term in the synchronizing sequence do you wish to generate?")
            print()
            auxVar = str(input("---> "))
            if auxVar == "back":
                synch_seq_interface()
            elif auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                auxVar = int(auxVar)
                automaton = synchronizing_sequence(automaton, auxVar)
                variable_check(automaton, "the " + str(auxVar) + "th term", synch_seq_interface)
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                synch_seq_interface()

def check_strong_synch_interface():
    print()
    print("Here you can find out whether or not an automaton / transducer is strongly")
    print("synchronizing.")
    print()
    print("What is the variable name of the automaton / transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            automaton = globals()[variableName]
            print()
            if IsStronglySynchronizing(automaton):
                print("The automaton / transducer is strongly synchronizing.")
            else:
                print("The automaton / transducer is NOT strongly synchronizing.")
            print()
            print("Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                synch_seq_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                synch_seq_interface()

def synch_level_interface():
    print()
    print("Here you can find, for an automaton / transducer, the minimal integer 'k'")
    print("such that the automaton / transducer is synchronizing at level k.")
    print()
    print("What is the variable name of the automaton / transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            automaton = globals()[variableName]
            print()
            if IsStronglySynchronizing(automaton):
                print("The minimal synchronizing level of the automaton / transducer is: ", min_synch_level(automaton))
            else:
                print("The automaton / transducer is NOT strongly synchronizing.")
            print()
            print("Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                synch_level_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                synch_level_interface()

def check_core_interface():
    print()
    print("Here you can find out whether or not a strongly synchronizing")
    print("automaton or transducer is core.")
    print()
    print("What is the variable name of the automaton or transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            if len(transd[0]) == 4:
                which = "transducer"
            else:
                which = "automaton"
            print()
            if not IsStronglySynchronizing(transd):
                print("The " + which + " isn't strongly synchronizing.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_core_interface()
            else:
                if IsCore(transd):
                    print("The " + which + " is core.")
                else:
                    print("The " + which + " is NOT core.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_core_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_core_interface()

def core_interface():
    print()
    print("Here you can generate the core of an automaton / transducer.")
    print("that is strongly synchronizing.")
    print()
    print("What is the variable name of the automaton / transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            if IsStronglySynchronizing(transducer):
                transducer = take_core(transducer)
                variable_check(transducer, "core", core_interface)
            else:
                print()
                print("The automaton / transducer is not strongly synchronizing.)")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    core_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                core_interface()

def check_synchronous_interface():
    print()
    print("Here you can find out whether or not a transducer is synchronous.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            print()
            if IsSynchronous(transd):
                print("The transducer is synchronous.")
            else:
                print("The transducer is NOT synchronous.")
            print("Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_synchronous_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_synchronous_interface()

def check_automata_inv_interface():
    print()
    print("Here you can find out whether or not a synchronous transducer")
    print("is invertible (in the automata-theoretic sense).")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            print()
            if not IsSynchronous(transd):
                print("The transducer isn't synchronous.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_automata_inv_interface()
            else:
                if IsInvertible(transd):
                    print("The transducer is invertible.")
                else:
                    print("The transducer is NOT invertible.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_automata_inv_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_automata_inv_interface()

def invert_synchronous_interface():
    print()
    print("Here you can generate the automata-theoretic inverse of")
    print("a synchronous transducer.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transducer = globals()[variableName]
            if IsSynchronous(transducer):
                if IsInvertible(transducer):
                    transducer = invert(transducer)
                    variable_check(transducer, "inverse", invert_synchronous_interface)
                else:
                    print()
                    print("The transducer is not invertible (in the automata-theoretic sense).")
                    print("Type anything to return to the previous menu.")
                    print()
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        invert_synchronous_interface()
            else:
                print()
                print("The transducer is not synchronous.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    invert_synchronous_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                invert_synchronous_interface()

def check_bisynch_interface():
    print()
    print("Here you can find out whether or not a synchronos")
    print("transducer is bi-synchronizing.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            print()
            if not IsSynchronous(transd):
                print("The transducer isn't synchronous.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_bisynch_interface()
            else:
                if IsBiSynchronizing(transd):
                    print("The transducer is bi-synchronizing.")
                else:
                    print("The transducer is NOT bi-synchronizing.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    check_bisynch_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_bisynch_interface()

def check_in_Hn_interface():
    print()
    print("Here you can find out whether or not a transducer is a representative")
    print("of an element of the group H_n, that is, if the transducer belongs to")
    print("an omega-equivalence class of transducer that is an element of H_n.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            boolean = IsInHn(transd)
            print()
            if boolean != "Half true":
                if boolean:
                    print("The transducer is the weakly minimal representative of an element of H_n.")
                else:
                    print("The transducer is NOT a representative of an element of H_n.")
            else:
                print("The transducer is the representative of an element of H_n, though")
                print("it is not weakly minimal.")
            print("Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_in_Hn_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                check_in_Hn_interface()

def decomp_interface():
    print()
    print("Here you can take a transducer is a representative of an element of the")
    print("group H_n, and run the decomposition algorithm of the one-sided case paper")
    print("to obtain a list of finite order elements of H_n whose product is the")
    print("inputted transducer.")
    print()
    print("What is the variable name of the transducer?")
    print()
    variableName = str(input("---> "))
    if variableName == "back":
        functions_interface()
    elif variableName == "main menu":
        main_menu()
    elif variableName != "exit":
        if variableName in globals():
            transd = globals()[variableName]
            print()
            if not IsInHn(transd):
                print("This transducer is not in H_n.")
                print("Type anything to return to the previous menu.")
                print()
                auxVar = str(input("---> "))
                if auxVar == "main menu":
                    main_menu()
                elif auxVar != "exit":
                    decomp_interface()
            else:
                decomplist = decompose_Hn_element(transd)
                print("The element has been decomposed into " + str(len(decomplist)) + " torsion elements.")
                print()
                print("Would you like to save the list consisting of the torsion elements")
                print("as a variable in this Python session?")
                print()
                auxVar1 = ""
                while auxVar1 not in ["yes", "no", "main menu", "back", "exit"]:
                    print("Type 'yes' or 'no'.")
                    auxVar1 = str(input("---> "))
                if auxVar1 == "main menu":
                    main_menu()
                elif auxVar1 == "back":
                    decomp_interface()
                elif auxVar1 != "exit":
                    if auxVar1 == "yes":
                        keepLooping = True
                        exitCondition = False
                        while keepLooping:
                            print()
                            print("What variable name do you wish the list of torsion elements to have?")
                            print()
                            variableName = str(input("---> "))
                            if variableName == "back":
                                keepLooping = False
                                decomp_interface()
                            elif variableName == "main menu":
                                keepLooping = False
                                main_menu()
                            elif variableName == "exit":
                                keepLooping = False
                            else:
                                if variableName in globals():
                                    print()
                                    print("'" + variableName + "' is currently assigned to a variable.")
                                    print("Are you sure you wish to overwrite this?")
                                    print()
                                    print("(type 'yes' or 'no')")
                                    print()
                                    auxVar3 = str(input("---> "))
                                    while auxVar3 != 'yes' and auxVar3 != 'no' and auxVar3 not in ["main menu", "back", "exit"]:
                                        print()
                                        print("Type 'yes' or 'no'")
                                        print()
                                        auxVar3 = str(input("---> "))
                                    if auxVar3 == "main menu":
                                        keepLooping = False
                                        main_menu()
                                    if auxVar3 == "exit":
                                        keepLooping = False
                                    if auxVar3 == "yes":
                                        keepLooping = False
                                        exitCondition = True
                                        globals()[variableName] = decomplist
                                        print("The list of torsion elements has been saved as the variable '" + variableName + "'.")
                                        auxVar1 = "no"
                                else:
                                    keepLooping = False
                                    exitCondition = True
                                    print()
                                    globals()[variableName] = decomplist
                                    print("The list of torsion elements has been saved as the variable '" + variableName + "'.")
                                if exitCondition:
                                    auxVar1 = "no"
                    if auxVar1 == "no":
                        print()
                        print("Would you like to generate a folder of .txt files of each torsion element?")
                        auxVar4 = ""
                        while auxVar4 not in ["yes", "no", "main menu", "back", "exit"]:
                            print("Type 'yes' or 'no'.")
                            auxVar4 = str(input("---> "))
                        if auxVar4 == "main menu":
                            main_menu()
                        elif auxVar4 == "back":
                            decomp_interface()
                        elif auxVar4 != "exit":
                            if auxVar4 == "yes":
                                keepLooping = True
                                exitCondition = False
                                while keepLooping:
                                    print()
                                    print("What name do you wish the folder to have?")
                                    folderName = str(input("---> "))
                                    if folderName == "back":
                                        keepLooping = False
                                        decomp_interface()
                                    elif folderName == "main menu":
                                        keepLooping = False
                                        main_menu()
                                    elif folderName == "exit":
                                        keepLooping = False
                                    else:
                                        try:
                                            os.mkdir(folderName)
                                            keepLooping = False
                                            exitCondition = True
                                        except FileExistsError:
                                            print()
                                            print("'" + folderName + "' is already a file in the directory.")
                                            print("Choose a different name, or delete the file and try again.")
                                if exitCondition:
                                    oldcwd = os.getcwd()
                                    newcwd = os.path.join(os.getcwd(), folderName)
                                    os.chdir(newcwd)
                                    string = "factor0"
                                    i = 0
                                    for transducer in decomplist:
                                        i = i + 1
                                        string = string[:-1]
                                        string = string + str(i)
                                        edgeList_to_txt(transducer, string)
                                    print()
                                    print("The folder of .txt files has been created.")
                                    os.chdir(oldcwd)
                                    auxVar4 = "no"
                            if auxVar4 == "no":
                                print()
                                print("Would you like to generate a folder of images of each torsion element?")
                                auxVar6 = ""
                                while auxVar6 not in ["yes", "no", "main menu", "back", "exit"]:
                                    print("Type 'yes' or 'no'.")
                                    auxVar6 = str(input("---> "))
                                if auxVar6 == "main menu":
                                    main_menu()
                                elif auxVar6 == "back":
                                    decomp_interface()
                                elif auxVar6 != "exit":
                                    if auxVar6 == "yes":
                                        keepLooping = True
                                        exitCondition = False
                                        while keepLooping:
                                            print()
                                            print("What name do you wish the folder to have?")
                                            folderName = str(input("---> "))
                                            if folderName == "back":
                                                keepLooping = False
                                                decomp_interface()
                                            elif folderName == "main menu":
                                                keepLooping = False
                                                main_menu()
                                            elif folderName == "exit":
                                                keepLooping = False
                                            else:
                                                try:
                                                    os.mkdir(folderName)
                                                    keepLooping = False
                                                    exitCondition = True
                                                except FileExistsError:
                                                    print()
                                                    print("'" + folderName + "' is already a file in the directory.")
                                                    print("Choose a different name, or delete the file and try again.")
                                        if exitCondition:
                                            oldcwd = os.getcwd()
                                            newcwd = os.path.join(os.getcwd(), folderName)
                                            os.chdir(newcwd)
                                            string = "factor0"
                                            i = 0
                                            for transducer in decomplist:
                                                i = i + 1
                                                string = string[:-1]
                                                string = string + str(i)
                                                draw_graph(transducer, string)
                                            print()
                                            print("The folder of .png files has been created.")
                                            os.chdir(oldcwd)
                                            print()
                                            print("Type anything to return to the previous menu.")
                                            print()
                                            auxVar8 = str(input("---> "))
                                            if auxVar8 == "main menu":
                                                main_menu()
                                            elif auxVar8 != "exit":
                                                decomp_interface()
        else:
            print()
            print("No such variable was found. Type anything to return to the previous menu.")
            print()
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                decomp_interface()

def Hn_product_interface():
    print()
    print("Here you can take two transducers that are elements of H_n,")
    print("and generate the transducer that is their product in H_n.")
    print()
    print("We are calculating transducer1 * transducer2")
    print()
    print("What is the variable name belonging to transducer1?")
    print()
    auxVar1 = str(input("---> "))
    if auxVar1 == "back":
        functions_interface()
    elif auxVar1 == "main menu":
        main_menu()
    elif auxVar1 != "exit":
        if auxVar1 in globals():
            transducer1 = globals()[auxVar1]
            if not IsInHn(transducer1):
                print()
                print("This transducer is not in H_n.")
                print("Type anything to return to the previous menu.")
                auxVar3 = str(input("---> "))
                if auxVar3 == "main menu":
                    main_menu()
                elif auxVar3 != "exit":
                    Hn_product_interface()
            print()
            print("What is the variable name belonging to transducer2?")
            print()
            auxVar2 = str(input("---> "))
            if auxVar2 == "back":
                Hn_product_interface()
            elif auxVar2 == "main menu":
                main_menu()
            elif auxVar2 != "exit":
                if auxVar2 in globals():
                    transducer2 = globals()[auxVar2]
                    if not IsInHn(transducer2):
                        print()
                        print("This transducer is not in H_n.")
                        print("Type anything to return to the previous menu.")
                        auxVar4 = str(input("---> "))
                        if auxVar4 == "main menu":
                            main_menu()
                        elif auxVar4 != "exit":
                            Hn_product_interface()
                    product = Hn_product(transducer1, transducer2)
                    variable_check(product, "product", Hn_product_interface)
                else:
                    print("No such variable found. Type anything to return to the previous menu.")
                    auxVar = str(input("---> "))
                    if auxVar == "main menu":
                        main_menu()
                    elif auxVar != "exit":
                        Hn_product_interface()
        else:
            print("No such variable found. Type anything to return to the previous menu.")
            auxVar = str(input("---> "))
            if auxVar == "main menu":
                main_menu()
            elif auxVar != "exit":
                Hn_product_interface()

def settings_interface():
    print()
    print("Welcome to the settings!")
    print()
    print("1) Change the directory for this program")
    print("2) Change the graphviz layout used for images of automata / transducers")
    print("3) Access the documentation for this program")
    print("4) Credits")
    print()
    auxVar = str(input("---> "))
    if auxVar == "main menu" or auxVar == "back":
        main_menu()
    elif auxVar == "1":
        directory_interface()
    elif auxVar == "2":
        layout_interface()
    elif auxVar == "3":
        documentation_interface()
    elif auxVar == "4":
        credits_interface()
    elif auxVar != "exit":
        settings_interface()

def directory_interface():
    print()
    print("Here you can specify a folder which this program can then use to")
    print("save .txt files containing an automaton / transducer, or to load")
    print("such a file from the folder into the memory of this Python session,")
    print("or finally to save .png images of automata / transducers.")
    print()
    print("Before this menu has been gone through, such things will default")
    print("to looking at the current working directory for Python,")
    print("which will likely be wherever this Python file is saved.")
    print()
    print("If you are unsure how to set this up, type 'help'.")
    print()
    print("The current directory for this program is")
    print(os.getcwd())
    print()
    print("Please enter the path to the directory that you wish to be used for")
    print("these purposes, e.g. /home/elliott/Desktop/TGI directory")
    print("(note the format for directories may be different for your OS, e.g.")
    print(" back slashes may be used instead).")
    print()
    auxVar = str(input("---> "))
    if auxVar == "back":
        settings_interface()
    elif auxVar == "main menu":
        main_menu()
    elif auxVar == "help":
        print()
        print("The directory is the folder that will be used to save and load files")
        print("relating to transducers. The best way to set this up is to open the")
        print("file explorer on your OS, and navigate to where you want to store the")
        print("the directory. Then, right click and click on 'Create new folder'. Give")
        print("the folder an appropriate name, right click on the folder and click on")
        print("'Properties'. There will be the path to the directory somewhere there.")
        print()
        print("Type anything to return to the previous menu, and type the path to the")
        print("directory into there.")
        print()
        auxVar = str(input("---> "))
        if auxVar == "main menu":
            main_menu()
        elif auxVar != "exit":
            directory_interface()
    elif auxVar != "exit":
        os.chdir(auxVar)
        print("Your directory has been saved as " + auxVar)
        print()
        print("Type anything to return to the previous menu.")
        print()
        auxVar1 = str(input("---> "))
        if auxVar1 == "back":
            directory_interface()
        elif auxVar1 != "exit":
            main_menu()

def layout_interface():
    try:
        with open("graphviz_layout.txt", "r") as data:
            currentlayout = data.readline()
    except FileNotFoundError:
        currentlayout = "dot"
    print()
    print("Here you can change the layout used by pygraphviz to generate")
    print("images of automata and transducers.")
    print()
    print("pygraphviz currently supports the following options.")
    print("For more information on these layouts, see")
    print("https://graphviz.gitlab.io/pdf/dot.1.pdf")
    print()
    print("The current layout is " + currentlayout + ".")
    print()
    print("Choose from the following:")
    print()
    print("1) neato")
    print("2) dot")
    print("3) twopi")
    print("4) circo")
    print("5) sfdp")
    print()
    auxVar1 = str(input("---> "))
    if auxVar1 == "main menu":
        main_menu()
    elif auxVar1 == "back":
        settings_interface()
    elif auxVar1 == "1":
        newlayout = "neato"
        with open("graphviz_layout.txt", "w") as data:
            data.write(newlayout)
    elif auxVar1 == "2":
        newlayout = "dot"
        with open("graphviz_layout.txt", "w") as data:
            data.write(newlayout)
    elif auxVar1 == "3":
        newlayout = "twopi"
        with open("graphviz_layout.txt", "w") as data:
            data.write(newlayout)
    elif auxVar1 == "4":
        newlayout = "circo"
        with open("graphviz_layout.txt", "w") as data:
            data.write(newlayout)
    elif auxVar1 == "5":
        newlayout = "sfdp"
        with open("graphviz_layout.txt", "w") as data:
            data.write(newlayout)
    if auxVar1 in [str(i) for i in range(1,6)]:
        print("The layout has been changed to " + newlayout)
        print("Type anything to return to the previous menu.")
        auxVar2 = str(input("---> "))
        if auxVar2 == "main menu":
            main_menu()
        elif auxVar2 != "exit":
            layout_interface()

def documentation_interface():
    print()
    print("The documentation for this software can be found at the following link:")
    print()
    print("https://github.com/elliottmaths/Transducer-Groups-Interface-documentation/blob/master/Transducer_Groups_Interface.pdf")
    print()
    print("Type anything to return to the previous menu.")
    auxVar = str(input('---> '))
    settings_interface()

def credits_interface():
    print()
    print("This program was written by Elliott Cawtheray, a mathematics undergraduate at the")
    print("University of St Andrews, during a summer research project that was funded")
    print("by the Edinburgh Mathematical Society, for which he is tremendously grateful.")
    print()
    print("The project was supervised by Dr. Collin Bleak - another recipient of the")
    print("author's gratitude. This software written under his supervision, with")
    print("algorithms implemented in this code either coming from his research")
    print("or from discussions with him.")
    print()
    print("Type anything to return to the previous menu.")
    auxVar = str(input('---> '))
    settings_interface()

if __name__ == "__main__":
    main_menu()